#' Realign a read to reference sequence
#'
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' @param reads.obj A read object
#' @param pathog.mut.obj A pathogenic mutation object
#' @param check.soft.clipping A logical value indicating whether soft-clipped reads to be realigned
#' @param softClippedReads.realign.match A non-negative integer specifying the scoring for a nucleotide match for realigning soft-clipped reads
#' @param softClippedReads.realign.mismatch A non-negative integer specifying the scoring for a nucleotide mismatch for realigning soft-clipped reads
#' @param softClippedReads.realign.gapOpening A non-negative integer specifying the cost for opening a gap in the realignment of soft-clipped reads
#' @param softClippedReads.realign.gapExtension A non-negative integer specifying the incremental cost incurred along the length of the gap in the realignment of soft-clipped reads
#' @param softClippedReads.realign.window.left A non-negative integer specifying the length of flanking region to be added to the left end of the original alignment region for realigning soft-clipped reads
#' @param softClippedReads.realign.window.right A non-negative integer specifying the length of flanking region to be added to the right end of the original alignment region for realigning soft-clipped reads
#' @param indel.shifting A character variable specifying the direction (left or right) for shifting the start position of indels till it is no longer possible to do so
#' 
#' @return A list of realigned results for a read including a flag of on-target reads, realigned read object, a pair of realigned read and ref sequences
#' 
#' @importFrom BSgenome getSeq
#' @importFrom GenomicRanges GRanges
#' @importFrom IRanges IRanges
#'
#' @examples
#' \dontrun{
#'     realignSingleRead()
#' }
#'
#' @noRd
realignSingleRead <- function(
        reference.genome,
        reads.obj,
        pathog.mut.obj, 	
        check.soft.clipping,
        softClippedReads.realign.match,
        softClippedReads.realign.mismatch,
        softClippedReads.realign.gapOpening,
        softClippedReads.realign.gapExtension,
        softClippedReads.realign.window.left,
        softClippedReads.realign.window.right,
        indel.shifting
	){
    
    chr.name <- pathog.mut.obj$chr.name
    pathog.mut.start <- pathog.mut.obj$mut.start
    pathog.mut.end <- pathog.mut.obj$mut.end
    
    is.scReads <- FALSE
	is.reads.target <- FALSE
	reads.obj.out <- reads.obj
	realn.seqs <- NA
	
	reads <- reads.obj$SEQ
	aln.pos <- reads.obj$POS
	cigar <- reads.obj$CIGAR
	cigar.trimmed <- trimCIGAR(cigar, reads, check.soft.clipping)
	cigar <- cigar.trimmed$cigar
	reads <- cigar.trimmed$reads
	ref.length <- cigar.trimmed$ref.length
	ref.start <- aln.pos
	ref.end <- ref.start + ref.length - 1
	
	non.dna.base.prop <- nchar(gsub("[ATCG]","",reads))/nchar(reads)
	
	if( non.dna.base.prop < 0.01 ){
	
    	if( grepl("S", cigar) ){
    	    is.scReads <- TRUE
    		cigar.scReads <- alignSoftClippedRead(
    			reads, 
    			reference.genome, 
    			chr.name, 
    			ref.start, 
    			ref.end, 
    			softClippedReads.realign.match,
    			softClippedReads.realign.mismatch,
    			softClippedReads.realign.gapOpening,
    			softClippedReads.realign.gapExtension,
    			softClippedReads.realign.window.left,
    			softClippedReads.realign.window.right,
    			indel.shifting
    		)
    		cigar <- cigar.scReads$cigar
    		ref.start <- cigar.scReads$ref.start
    		ref.end <- cigar.scReads$ref.end
    		ref.length <- ref.end - ref.start + 1
    		aln.pos <- ref.start
    	}
    	
    	if( ref.start <= pathog.mut.start & ref.end >= pathog.mut.end ){
    	    is.reads.target <- TRUE
    		ref <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(ref.start,ref.end)) ) )
    		
            realn <- alignReadToRef(
                reference.genome,
                pathog.mut.obj, 
                ref,
                ref.start,
                ref.end,
                reads, 
                is.scReads,
                softClippedReads.realign.match,
                softClippedReads.realign.mismatch,
                softClippedReads.realign.gapOpening,
                softClippedReads.realign.gapExtension,
                indel.shifting
            )
            
            cigar.trimmed2 <- trimCIGAR(realn$cigar, realn$reads, keep.soft.clipping=FALSE)
            
            ref <- realn$ref 
            ref.start <- realn$ref.start
            ref.end <- realn$ref.end
            reads <- cigar.trimmed2$reads
            realn.cigar <- cigar.trimmed2$cigar
    		aln.pos <- realn$ref.start
    		
    		reads.obj.out$CIGAR <- paste0(
    		    cigar.trimmed$cigar.trimmed.left, 
    		    cigar.trimmed2$cigar.trimmed.left, 
    		    realn.cigar, 
    		    cigar.trimmed2$cigar.trimmed.right,
    		    cigar.trimmed$cigar.trimmed.right
    		)
    		reads.obj.out$POS <- aln.pos
    		realn.df <- decodeCIGAR(realn.cigar, ref, reads)
    		realn.seqs <- c(
    		    paste(realn.df$ref_seq, collapse = ""),
    		    paste(realn.df$reads_seq, collapse = "")
    		)
    	}
	    
	}
	
	res <- list(
		is.reads.target = is.reads.target,
		realigned.reads.obj = reads.obj.out,
		realigned.sequences = realn.seqs
	)
	return(res)
}
