#include <Rcpp.h>
#include <cmath>
#include <boost/math/quadrature/trapezoidal.hpp>
using namespace Rcpp;
// [[Rcpp::depends(BH)]]


double con_pdf(double y, NumericVector x, NumericVector theta) {
  int p = theta.size();
  
  NumericVector beta = theta[Range(0, p - 3)];
  double sigma2 = theta[p - 2];
  double lambda = theta[p - 1];
  
  double beta_x = 0.0;
  for (int i = 0; i < beta.size(); i++) {
    beta_x += beta[i] * x[i];
  }
  
  double phi = R::dnorm(y, beta_x, sqrt(sigma2), false);
  double Phi = R::pnorm(-(y - beta_x) * lambda / sqrt(sigma2), 0.0, 1.0, true, false);
  
  return 2 * phi * Phi;
}

/*
#include <boost/math/quadrature/gauss_kronrod.hpp> 
 // [[Rcpp::export]]
 double integral_f2_cpp(double alpha, double lambda) {
 
 auto integrand = [alpha, lambda](double z) {
 double exp_term = std::exp(-(1.0 + alpha) * z * z / 2.0);
 double pnorm_term = 0.5 * (1.0 + std::erf(-z * lambda / std::sqrt(2.0)));
 return exp_term * std::pow(pnorm_term, 1.0 + alpha);
 };
 
 double result;
 double tol = 1e-12; 
 result = boost::math::quadrature::gauss_kronrod<double, 101>::integrate(
 integrand, -std::numeric_limits<double>::infinity(), std::numeric_limits<double>::infinity(), tol
 );
 
 return result;
 }
 */


// [[Rcpp::export]]
double integral_f(double alpha, double lambda) {
  if(alpha==1){
    return std::atan(std::sqrt(1 + lambda * lambda)) / std::sqrt(M_PI);
  }
  else{
    auto integrand = [alpha, lambda](double t) {
      double z = std::tan(t);
      double dz_dt = 1.0 / (std::cos(t) * std::cos(t));
      double exp_term = std::exp(-(1.0 + alpha) * z * z / 2.0);
      double pnorm_term =R::pnorm(-z * lambda, 0.0, 1.0, true, false);
      return exp_term * std::pow(pnorm_term, 1.0 + alpha) * dz_dt;
    };
    
    double tol = 1e-12;
    double result = boost::math::quadrature::trapezoidal(integrand, -M_PI_2, M_PI_2, tol);
    
    return result;
  }
}



// [[Rcpp::export]]
double H_alpha(double y, NumericVector x, NumericVector theta, double alpha) {
  int p = theta.size();
  NumericVector beta = theta[Range(0, p-3)];
  double sigma2 = theta[p-2];
  double lambda = theta[p-1];
  
  double f_yx = con_pdf(y, x, theta);
  
  if (alpha == 0) {
    return -log(f_yx);
  } else {
    double int_f = integral_f(alpha, lambda);
    double term1 = pow(2 / M_PI, (1 + alpha) / 2) / pow(sigma2, alpha / 2) * int_f;
    double term2 = (1 + 1 / alpha) * pow(f_yx, alpha);
    
    return term1 - term2;
  }
}


double H0_alpha(double y, NumericVector x, NumericVector beta, 
                    double sigma2, double lambda, double alpha) {
  double g_x_beta = 0.0;
  for (int i = 0; i < x.size(); i++) {
    g_x_beta += beta[i] * x[i];
  }
  
  double z = (y - g_x_beta) / sqrt(sigma2);

  double term1 = (1.0 + 1.0 / alpha);
  double term2 = exp(-alpha * z * z / 2.0);
  double term3 = pow(R::pnorm(-z * lambda, 0.0, 1.0, true, false), alpha);
  
  return term1 * term2 * term3;
}






// [[Rcpp::export]]
double obj_f0_cpp(NumericVector params, NumericVector y, NumericMatrix X) {
  int n = y.size(); 
  double result = 0.0; 
  
  for (int i = 0; i < n; i++) {
    NumericVector x_row = X(i, _); 
    double pdf_value = con_pdf(y[i], x_row, params); 
      result += log(pdf_value); 
  }
  
  return result; 
}
  
  
// [[Rcpp::export]]
double obj_f_alpha_cpp(NumericVector params, NumericVector y, NumericMatrix X, double alpha) {
  int n = y.size();
  int n_par = params.size();
  
  NumericVector beta = params[Range(0, n_par - 3)];
  double sigma2 = params[n_par - 2];
  double lambda = params[n_par - 1];
  
  double term1 = sqrt(2) / sqrt(M_PI) * integral_f(alpha, lambda);
  
  double term2 = 0.0;
  for (int i = 0; i < n; i++) {
    NumericVector x_row = X(i, _);
    term2 += H0_alpha(y[i], x_row, beta, sigma2, lambda, alpha);
  }
  
  double result = (term1 - term2 / double(n)) / pow(sigma2, alpha / 2.0);
  return result;
}

