% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/extract_saccades.R
\name{extract_saccades}
\alias{extract_saccades}
\title{Extract saccades from samples using votes from selected methods.}
\usage{
extract_saccades(
  x,
  y,
  sample_rate,
  trial = NULL,
  methods = list(method_ek, method_om, method_nh),
  velocity_function = saccadr::diff_ek,
  options = NULL,
  binocular = "merge",
  vote_threshold = ifelse(length(methods) == 1, 1, (length(methods) - 1)),
  minimal_duration_ms = 12,
  minimal_separation_ms = 12,
  return_votes = FALSE
)
}
\arguments{
\item{x}{Horizontal coordinate, either a vector for monocular data or a two-column matrix for binocular data.}

\item{y}{Vertical coordinate, either a vector for monocular data or a two-column matrix for binocular data.}

\item{sample_rate}{Sampling rate in Hz. It is assumed to be common for the entire time series.
If the time series contains chunks (trials) that were recorded using different acquisition rate
(e.g., SR Research Eyelink allows to set different acquisition rate for each recording / trial),
you would need to split the time series and analyze them separately.}

\item{trial}{Optional vector with trial ID. If omitted, all samples are assumed to belong to a single trial.
Velocity, acceleration, and saccades themselves are computed respecting trial borders.}

\item{methods}{A list with saccade detection methods, can include external functions that
implement sample classification (see \emph{Using Custom Methods} vignette). Package methods include
Engbert & Kliegl (2003) (\code{method_ek}), Otero-Millan et al. (2014) (\code{method_om}), 
Nyström and Holmqvist (2010) (\code{method_nh}). Defaults to the list of all internally implemented
methods: \code{list(method_ek, method_om, method_nh)}.}

\item{velocity_function}{A handle to a function to compute velocity and acceleration. Defaults to a method
suggested by Engbert & Kliegl (2003) \code{\link{diff_ek}}. The package also implements the method proposed
by Nyström and Holmqvist (2010) \code{\link{diff_nh}}. See vignette "Velocity computation" for details and
information on how to implement a custom method.}

\item{options}{A named list with options for saccade detection (see \code{\link{method_ek}}, \code{\link{method_om}},
\code{\link{method_nh}}) and velocity (\code{\link{diff_ek}}, \code{\link{diff_nh}}) computation.
See documentation on specific method for details.}

\item{binocular}{Specifies how a binocular data is treated. Options are \code{"cyclopean"} (binocular data is
converted to an average cyclopean image before saccades are extracted), \code{"monocular"} (saccades
are extracted independently for each eye), \code{"merge"} (default, sample votes are obtained from both eyes and
for all methods and then averaged. This way only binocular saccades, i.e., eye movements with a sufficient temporal
 overlap between eyes, are detected.).}

\item{vote_threshold}{Value between 1 and N (where N is number of used methods) defining a vote threshold for a
saccade. By default, all but one method \eqn{threshold = N-1} must agree for a sample to be considered for a saccade.
Threshold of 1 is applied if a single method is used.}

\item{minimal_duration_ms}{Minimal duration of a saccade in milliseconds. Shorter candidate saccades are discarded,}

\item{minimal_separation_ms}{Minimal time separation between saccades in milliseconds. Saccades that are
separated by a shorter interval of "not a saccade" votes, will be merged including that period.}

\item{return_votes}{Logical. Whether function should return extracted microsaccades (\code{FALSE}, default)
or votes per sample (\code{TRUE}).}
}
\value{
A \code{data.frame} with saccade properties (see \strong{details}), if \code{return_votes = FALSE}.
Alternatively, it returns votes per sample (\code{return_votes = TRUE}). For a monocular processing (monocular
input, cyclopean or merged binocular data) it is a matrix with \code{nrow(x)} rows and \code{length(methods)}
 columns with 0/1 votes for each sample and method. For binocular processing, function returns a two element \code{list} with
 the similar matrices but per eye.
}
\description{
Extract saccades from samples using votes from selected methods. Each method votes whether
a given sample belongs to a saccade. Next, saccades are identified via a majority vote using the 
\code{vote_threshold} parameter, as well as a minimum duration and minimal temporal separation criteria.
Please note that units of the gaze samples must be in  \strong{degrees of visual angle}. The units are important
as some methods use specific (e.g., physiologically plausible) velocity and acceleration thresholds.
 
By default, ensemble includes methods proposed by Engbert & Kliegl (2003) (\code{"ek"}),
Otero-Millan et al. (\code{"om"}), and Nyström & Holmqvist (2010) (\code{"nh"}), 
see \emph{Implemented Methods} vignette. However, it can be extended
via custom methods, see \emph{Using Custom Methods} vignette.

By default, the function returns a table with identified saccades but can return a matrix with methods' votes
per sample instead (\code{return_votes = TRUE}).
}
\details{
Variables that describe saccade
\itemize{
\item{\code{Trial}} Trial index.
\item{\code{Eye}} {\code{"Monocular"} for monocular inputs. \code{"Cyclopean"} for binocular data that
was averaged \emph{before} applying algorithms. \code{"Binocular"} for binocular data with votes
averaged \emph{after} applying algorithms. \code{"Left"} or \code{"Right"} for binocular data
when eyes are processed independently.}
\item{\code{OnsetSample}} {Index of the first sample.}
\item{\code{OffsetSample}} {Index of the last sample.}
\item{\code{Onset}} {Onset time relative to the trial start in milliseconds.}
\item{\code{Offset}} {Offset time relative to the trial start in milliseconds.}
\item{\code{Duration}} {Duration in milliseconds.}
\item{\code{DisplacementX}} {Horizontal displacement measured from the \emph{first} to the \emph{last} sample.}
\item{\code{DisplacementY}} {Vertical displacement measured from the \emph{first} to the \emph{last} sample.}
\item{\code{Displacement}} {Displacement magnitude measured from the \emph{first} to the \emph{last} sample.}
\item{\code{DisplacementPhi}} {Displacement direction measured from the \emph{first} to the \emph{last} sample.}
\item{\code{AmplitudeX}} {Horizontal displacement measured from the \emph{leftmost} to the \emph{rightmost} sample.}
\item{\code{AmplitudeY}} {Vertical displacement measured from the \emph{lowest} to the \emph{uppermost} sample.}
\item{\code{Amplitude}} {Displacement magnitude measured from the most extreme samples.}
\item{\code{AmplitudePhi}} {Displacement direction measured from the most extreme samples.}
\item{\code{VelocityPeak}} {Peak velocity.}
\item{\code{VelocityAvg}} {Average velocity.}
\item{\code{AccelerationPeak}} {Peak acceleration.}
\item{\code{AccelerationAvg}} {Average acceleration.}
\item{\code{AccelerationStart}} {Peak acceleration \emph{before} peak velocity was reached.}
\item{\code{AccelerationStop}} {Peak acceleration \emph{after} peak velocity was reached.}
}
}
\examples{
# Single trial
data(single_trial)
saccades <- extract_saccades(single_trial$x, single_trial$y, 500)

# Multiple trials
data(monocular_ten_trials)
saccades <- extract_saccades(monocular_ten_trials$x,
                             monocular_ten_trials$y, 
                             500,
                             trial = monocular_ten_trials$trial)
 
 # binocular saccades                            
 data("single_trial_binocular")
 saccades_b <- saccadr::extract_saccades(single_trial_binocular[, c('xL', 'xR')],
                                         single_trial_binocular[, c('yL', 'yR')],
                                         sample_rate = 1000)
                                         
 # cyclopean saccades from binocular data
saccades_c <- saccadr::extract_saccades(single_trial_binocular[, c('xL', 'xR')],
                                        single_trial_binocular[, c('yL', 'yR')],
                                        sample_rate = 1000,
                                        binocular = "cyclopean")

 # monocular saccades from binocular data
saccades_m <- saccadr::extract_saccades(single_trial_binocular[, c('xL', 'xR')],
                                       single_trial_binocular[, c('yL', 'yR')],
                                       sample_rate = 1000,
                                       binocular = "monocular")
                             
# Using a single method
saccades <- extract_saccades(single_trial$x, single_trial$y, 500, methods = method_om)

# Using two methods
saccades <- extract_saccades(single_trial$x,
                             single_trial$y,
                             500,
                             methods = list(method_ek, method_om))

#  Alternative velocity computation method
saccades <- extract_saccades(single_trial$x, single_trial$y, 500, velocity_function = diff_nh)

# A strict unanimous decision threshold
saccades <- extract_saccades(single_trial$x, single_trial$y, 500, vote_threshold = 3)

# A slacker criterion that at least one of the three methods must label sample as a saccade
saccades <- extract_saccades(single_trial$x, single_trial$y, 500, vote_threshold = 1)

# Only longish saccades are extracted
saccades <- extract_saccades(single_trial$x, single_trial$y, 500, minimal_duration_ms = 20)
}
\seealso{
\code{\link{method_ek}}, \code{\link{method_om}}, \code{\link{method_nh}}, \code{\link{diff_ek}}, \code{\link{diff_nh}}
}
