% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/satdad_Rfunctions.R
\name{tsicEmp}
\alias{tsicEmp}
\title{Empirical tail superset importance coefficients.}
\usage{
tsicEmp(sample, ind = 2, k, sobol = FALSE, norm = FALSE)
}
\arguments{
\item{sample}{A \code{(n times d)} matrix.}

\item{ind}{A character string among "with.singletons" and "all" (without singletons), or an integer in \eqn{\{2,...,d\}} or a list of subsets from  \eqn{\{1,...,d\}}. The default is \code{ind = 2}, all pairwise coefficients are computed.}

\item{k}{An integer smaller or equal to \code{n}.}

\item{sobol}{A boolean. `FALSE` (the default). If `TRUE`:  the index is normalized by the empirical global variance.}

\item{norm}{A boolean. `FALSE` (the default). If `TRUE`: the index is normalized by its theoretical upper bound.}
}
\value{
The function returns a list of two elements:

\itemize{
\item{\code{subsets}} A list of subsets from  \eqn{\{1,...,d\}}.

When \code{ind} is given as an integer, \code{subsets} is the list of subsets from  \eqn{\{1,...,d\}} with cardinality \code{ind}. When \code{ind} is the list, it corresponds to \code{subsets}.

When \code{ind = "with.singletons"}  subsets is the list of all non empty subsets in \eqn{\{1,...,d\}}.

When \code{ind = "all"}   subsets is the list of all subsets in \eqn{\{1,...,d\}} with cardinality larger or equal to 2.

\item{\code{tsic}} A vector of empirical tail superset importance coefficients associated with the list \code{subsets}. When \code{norm = TRUE}, then tsic  are normalized in the sense that the original values are divided by corresponding upper bounds.
}
}
\description{
Computes on a sample the tail superset importance coefficients (tsic) associated with threshold \code{k}. The value may be renormalized by the empirical global variance (Sobol version) and/or by its theoretical upper bound.
}
\details{
The theoretical functional decomposition of the variance of the stdf \eqn{\ell} consists in writing \eqn{D(\ell) = \sum_{I \subseteq \{1,...,d\}} D_I(\ell) } where \eqn{D_I(\ell)} measures the variance of \eqn{\ell_I(U_I)} the term associated with subset \eqn{I} in the Hoeffding-Sobol decomposition of \eqn{\ell}
; note that \eqn{U_I} represents a random vector with independent standard uniform entries.

Fixing a subset of components \eqn{I}, the theoretical tail superset importance coefficient is defined by \eqn{\Upsilon_I(\ell)=\sum_{J \supseteq I} D_J(\ell)}.
A theoretical upper bound for tsic \eqn{\Upsilon_I(\ell)} is given by Theorem 2 in Mercadier and Ressel (2021)
which states that \eqn{\Upsilon_I(\ell)\leq 2(|I|!)^2/((2|I|+2)!)}.

Here, the function \code{tsicEmp} evaluates, on a \eqn{n}-sample and threshold \eqn{k},  the empirical tail superset  importance coefficient \eqn{\hat{\Upsilon}_{I,k,n}} the empirical counterpart of \eqn{\Upsilon_I(\ell)}.

Under the option \code{sobol = TRUE}, the function \code{tsicEmp} returns  \eqn{\dfrac{\hat{\Upsilon}_{I,k,n}}{\hat{D}_{k,n}}} the empirical counterpart of \eqn{\dfrac{\Upsilon_I(\ell)}{D_I(\ell)}}.

Under the option \code{norm = TRUE}, the quantities are multiplied by \eqn{\dfrac{(2|I|+2)!}{2(|I|!)^2}}.

Proposition 1 and Theorem 2 of Mercadier and Roustant (2019) provide several rank-based expressions

\eqn{\hat{\Upsilon}_{I,k,n}=\frac{1}{k^2}\sum_{s=1}^n\sum_{s^\prime=1}^n \prod_{t\in I}(\min(\overline{R}^{(t)}_s,\overline{R}^{(t)}_{s^\prime})-\overline{R}^{(t)}_{s}\overline{R}^{(t)}_{s^\prime}) \prod_{t\notin I} \min(\overline{R}^{(t)}_s,\overline{R}^{(t)}_{s^\prime})}

\eqn{\hat{D}_{k,n}=\frac{1}{k^2}\sum_{s=1}^n\sum_{s^\prime=1}^n \prod_{t\in I}\min(\overline{R}^{(t)}_s,\overline{R}^{(t)}_{s^\prime})- \prod_{t\in I}\overline{R}^{(t)}_{s}\overline{R}^{(t)}_{s^\prime}}

where

\itemize{
\item{} \eqn{k} is the threshold parameter,
\item{} \eqn{n} is the sample size,
\item{} \eqn{X_1,...,X_n} describes the \code{sample}, each \eqn{X_s} is a d-dimensional vector \eqn{X_s^{(t)}} for \eqn{t=1,...,d},
\item{} \eqn{R^{(t)}_s} denotes the rank of \eqn{X^{(t)}_s} among \eqn{X^{(t)}_1, ..., X^{(t)}_n},
\item{} and  \eqn{\overline{R}^{(t)}_s = \min((n- R^{(t)}_s+1)/k,1)}.
}
}
\examples{

## Fix a 6-dimensional asymmetric tail dependence structure
ds <- gen.ds(d = 6, sub = list(1:4,5:6))

## Plot the  tail dependograph
graphs(ds)

## Generate a 1000-sample of Archimax Mevlog random vectors
## associated with ds and underlying distribution exp
sample <- rArchimaxMevlog(n = 1000, ds = ds, dist = "exp", dist.param = 1.3)

## Compute tsic values associated with subsets
## of cardinality 2 or more \code{ind = "all"}
res <- tsicEmp(sample = sample, ind = "all", k = 100, sobol = TRUE, norm = TRUE)

## Select the significative tsic
indices_nonzero <- which(res$tsic \%in\% boxplot.stats(res$tsic)$out == TRUE)

## Subsets associated with significative tsic reflecting the tail support
as.character(res$subsets[indices_nonzero])

## Pairwise tsic are obtained by
res_pairs <- tsicEmp(sample = sample, ind = 2, k = 100, sobol = TRUE, norm = TRUE)

## and plotted in the tail dependograph
graphsEmp(sample, k = 100)
}
\references{
Mercadier, C. and Ressel, P. (2021)
Hoeffding–Sobol decomposition of homogeneous co-survival functions: from Choquet representation to extreme value theory application.
Dependence Modeling, \bold{9(1)}, 179--198.

Mercadier, C. and Roustant, O. (2019)
The tail dependograph.
\emph{Extremes}, \bold{22}, 343--372.
}
\seealso{
\code{\link[satdad]{graphsEmp}}, \code{\link[satdad]{ellEmp}}
}
\author{
Cécile Mercadier (\code{mercadier@math.univ-lyon1.fr})
}
