% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scf_percentile.R
\name{scf_percentile}
\alias{scf_percentile}
\title{Estimate Percentiles in SCF Microdata}
\usage{
scf_percentile(scf, var, q = 0.5, by = NULL, verbose = FALSE)
}
\arguments{
\item{scf}{A \code{scf_mi_survey} object created with \code{\link[=scf_load]{scf_load()}}. Must
contain the list of replicate-weighted designs for each implicate in
\code{scf$mi_design}.}

\item{var}{A one-sided formula naming the continuous variable to
summarize (for example \code{~networth}).}

\item{q}{Numeric percentile in between 0 and 1. Default 0.5 (median).}

\item{by}{Optional one-sided formula naming a categorical grouping
variable. If supplied, the percentile is estimated separately within
each group.}

\item{verbose}{Logical. If TRUE, include implicate-level estimates in
the returned object for inspection. Default FALSE.}
}
\value{
An object of class \code{"scf_percentile"} containing:
\describe{
\item{results}{A data frame containing pooled percentile estimates, pooled
standard errors, and implicate min/max values. One row per group (if
\code{by} is supplied) or one row otherwise.}
\item{imps}{A list of implicate-level percentile estimates and standard errors.}
\item{aux}{A list containing the variable name, optional group variable name,
and the quantile requested.}
\item{verbose}{Logical flag indicating whether implicate-level estimates
should be printed by \code{print()} or \code{summary()}.}
}
}
\description{
This function estimates a weighted percentile of a continuous variable
in the Survey of Consumer Finances (SCF). It reproduces the procedure used
in the Federal Reserve Board's published SCF Bulletin SAS macro for
distributional statistics (Federal Reserve Board 2023c). This convention is
specific to SCF descriptive distributional statistics (quantiles,
proportions) and differs from standard handling (i.e., using Rubin's Rule).
}
\details{
The operation to render the estimates:
\enumerate{
\item For each implicate, estimate the requested percentile using
\code{\link[survey:svyquantile]{survey::svyquantile()}} with \code{se = TRUE}.
\item The reported point estimate is the mean of the M implicate-specific
percentile estimates.
\item The standard error follows the SCF Bulletin SAS macro convention:

\if{html}{\out{<div class="sourceCode">}}\preformatted{V_total = V1 + ((M + 1) / M) * B
}\if{html}{\out{</div>}}

where:
\itemize{
\item V1 is the replicate-weight sampling variance of the percentile
from the first implicate only.
\item B  is the between-implicate variance of the percentile estimates.
}

The reported standard error is sqrt(V_total).
\item If a grouping variable is supplied, the same logic is applied
separately within each group.
}
}
\examples{
# Do not implement these lines in real analysis:
# Use functions `scf_download()` and `scf_load()` for actual SCF data
td <- tempfile("percentile_")
dir.create(td)

src <- system.file("extdata", "scf2022_mock_raw.rds", package = "scf")
file.copy(src, file.path(td, "scf2022.rds"), overwrite = TRUE)
scf2022 <- scf_load(2022, data_directory = td)

# Estimate the 75th percentile of net worth
scf_percentile(scf2022, ~networth, q = 0.75)

# Estimate the median net worth by ownership group
scf_percentile(scf2022, ~networth, q = 0.5, by = ~own)

# Do not implement these lines in real analysis: Cleanup for package check
unlink(td, recursive = TRUE, force = TRUE)
rm(scf2022)

}
\references{
Federal Reserve Board. 2023c. "SAS Macro: Variable Definitions."
https://www.federalreserve.gov/econres/files/bulletin.macro.txt
}
\seealso{
\code{\link[=scf_median]{scf_median()}}, \code{\link[=scf_mean]{scf_mean()}}
}
