% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sensitivity_functions.R
\name{benchmark_parameters}
\alias{benchmark_parameters}
\title{Helper function for benchmarking for sensitivity parameters}
\usage{
benchmark_parameters(
  weights,
  weights_benchmark,
  k_sigma = 1,
  k_rho = 1,
  Y,
  Z,
  sigma2,
  estimand = "ATT"
)
}
\arguments{
\item{weights}{Vector of estimated weights}

\item{weights_benchmark}{Vector of estimated weights, omitting the covariate (or set of covariates) being used to benchmark}

\item{k_sigma}{Relative ability of omitted confounder to explain variation in the true weights. If \code{k_sigma > 1}, then we expect the omitted confounder to explain more variation in the true weights than the benchmarked covariate(s). If \code{k_sigma < 1}, then we expect the omitted confounder to explain less of the variation in the true weights than the benchmarked covariate(s). Default is set to 1.}

\item{k_rho}{Relative correlation of omitted confounder with the outcome. If \code{k_rho > 1}, then we expect the omitted confounder to be more correlated with the outcome than the benchmarked covariate(s). If \code{k_rho < 1}, then we expect the omitted confounder to be less correlated with the outcome than the benchmarked covariate(s). Default is set to 1.}

\item{Y}{Vector of outcomes}

\item{Z}{Vector of treatment assignment (Only necessary if \code{estimand = "PATE"} in order to estimate covariances)}

\item{sigma2}{Estimated variance of the outcome (i.e., stats::var(Y) for obervational setting; stats::var(tau) for generalization setting)}

\item{estimand}{String specifying the estimand of interest. Valid inputs are "PATE", "Augmented", "ATT", or "Survey".}
}
\value{
\code{data.frame} containing estimated parameter values for a confounder with specified relative confounder strength to an observed covariate (or set of covariates), as well as the estimated bias from such an omitted confounder.
}
\description{
Helper function that returns parameter estimates for an omitted variable with specified relative confounding strength to an observed covariate (or set of covariates)
}
\examples{
data(jtpa_women)
site_name <- "NE"
df_site <- jtpa_women[which(jtpa_women$site == site_name), ]
df_else <- jtpa_women[which(jtpa_women$site != site_name), ]

# Estimate unweighted estimator:
model_dim <- estimatr::lm_robust(Y ~ T, data = df_site)
PATE <- coef(lm(Y ~ T, data = df_else))[2]
DiM <- coef(model_dim)[2]
# Generate weights using observed covariates:
df_all <- jtpa_women
df_all$S <- ifelse(jtpa_women$site == "NE", 1, 0)
model_ps <- WeightIt::weightit(
  (1 - S) ~ . - site - T - Y, 
  data = df_all, method = "ebal", estimand = "ATT"
)
weights <- model_ps$weights[df_all$S == 1]
# Estimate IPW model:
model_ipw <- estimatr::lm_robust(Y ~ T, data = df_site, weights = weights)
ipw <- coef(model_ipw)[2]
# Estimate bound for var(tau):
vartau <- var(df_site$Y[df_site$T == 1]) - var(df_site$Y[df_site$T == 0])
RV <- robustness_value(estimate = ipw, b_star = 0, sigma2 = vartau, weights = weights)

# Select weighting variables:
weighting_vars <- names(df_all)[which(!names(df_all) \%in\% c("site", "S", "Y", "T"))]

# Run benchmarking:
df_benchmark <- run_benchmarking(
  weighting_vars = weighting_vars,
  data = df_all[, -1],
  treatment = "T", outcome = "Y", selection = "S",
  estimate = ipw,
  RV = RV, sigma2 = vartau,
  estimand = "PATE"
)

print(df_benchmark)

}
\keyword{internal}
