% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hotspot_kde.R
\name{hotspot_kde}
\alias{hotspot_kde}
\title{Estimate two-dimensional kernel density of points}
\usage{
hotspot_kde(
  data,
  cell_size = NULL,
  grid_type = "rect",
  bandwidth = NULL,
  bandwidth_adjust = 1,
  grid = NULL,
  weights = NULL,
  transform = TRUE,
  quiet = FALSE,
  ...
)
}
\arguments{
\item{data}{\code{\link[sf]{sf}} data frame containing points.}

\item{cell_size}{\code{numeric} value specifying the size of each equally
spaced grid cell, using the same units (metres, degrees, etc.) as used in
the \code{sf} data frame given in the \code{data} argument. Ignored if
\code{grid} is not \code{NULL}. If this argument and \code{grid} are
\code{NULL} (the default), the cell size will be calculated automatically
(see Details).}

\item{grid_type}{\code{character} specifying whether the grid should be made
up of squares (\code{"rect"}, the default) or hexagons (\code{"hex"}).
Ignored if \code{grid} is not \code{NULL}.}

\item{bandwidth}{\code{numeric} value specifying the bandwidth to be used in
calculating the kernel density estimates. If this argument is \code{NULL}
(the default), the bandwidth will be determined automatically using the
result of \code{\link[MASS]{bandwidth.nrd}} called on the co-ordinates of
\code{data}.}

\item{bandwidth_adjust}{single positive \code{numeric} value by which the
value of \code{bandwidth} is multiplied. Useful for setting the bandwidth
relative to the default.}

\item{grid}{\code{\link[sf]{sf}} data frame containing polygons, which will
be used as the grid for which densities are estimated.}

\item{weights}{\code{NULL} or the name of a column in \code{data} to be used
as weights for weighted counts and KDE values.}

\item{transform}{the underlying SpatialKDE package cannot calculate kernel
density for lon/lat data, so this must be transformed to use a projected
co-ordinate reference system. If this argument is \code{TRUE} (the
default) and \code{sf::st_is_longlat(data)} is \code{TRUE}, \code{data}
(and \code{grid} if provided) will be transformed automatically using
\code{link{st_transform_auto}} before the kernel density is estimated and
transformed back afterwards. Set this argument to \code{FALSE} to suppress
automatic transformation of the data.}

\item{quiet}{if set to \code{TRUE}, messages reporting the values of any
parameters set automatically will be suppressed. The default is
\code{FALSE}.}

\item{...}{Further arguments passed to \code{\link[SpatialKDE]{kde}}.}
}
\value{
An \code{\link[sf]{sf}} tibble of grid cells with corresponding point
counts and kernel density estimates for each cell. This can be plotted
using \code{\link{autoplot}}.
}
\description{
Estimate two-dimensional kernel density of points
}
\details{
This function creates a regular two-dimensional grid of cells (unless a
custom grid is specified with \code{grid}) and calculates the density of
points in each cell on that grid using functions from the \code{SpatialKDE}
package. The count of points in each cell is also returned.
\subsection{Coverage of the output data}{

The grid produced by this function covers the convex hull of the input data
layer. This means the result may include KDE values for cells that are
outside the area for which data were provided, which could be misleading. To
handle this, consider cropping the output layer to the area for which data
are available. For example, if you only have crime data for a particular
district, crop the output dataset to the district boundary using
\code{\link[sf]{st_intersection}}.
}

\subsection{Automatic cell-size selection}{

If no cell size is given then the cell size will be set so that there are 50
cells on the shorter side of the grid. If the \code{data} SF object is projected
in metres or feet, the number of cells will be adjusted upwards so that the
cell size is a multiple of 100.
}
}
\examples{
library(sf)

# Transform data to UTM zone 15N so that cell_size and bandwidth can be set
# in metres
memphis_robberies_utm <- st_transform(memphis_robberies_jan, 32615)

# Automatically set grid-cell size, bandwidth and neighbour distance
\donttest{
hotspot_kde(memphis_robberies_utm)
}

# Manually set grid-cell size and bandwidth in metres, since the
# `memphis_robberies_utm` dataset uses a co-ordinate reference system (UTM
# zone 15 north) that is specified in metres
\donttest{
hotspot_kde(memphis_robberies_utm, cell_size = 200, bandwidth = 1000)
}

}
\references{
Yin, P. (2020). Kernels and Density Estimation. \emph{The Geographic
Information Science & Technology Body of Knowledge} (1st Quarter 2020
Edition), John P. Wilson (ed.).
doi:\doi{10.22224/gistbok/2020.1.12}
}
