% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/powerShape.R
\name{powerShape}
\alias{powerShape}
\alias{tylerShape}
\alias{classicShape}
\title{M-estimators of Shape from the Power Family.}
\usage{
powerShape(x, alpha, center = NULL,
    normalization = c("det", "trace", "one"), maxiter = 1e4, eps = 1e-6)

tylerShape(x, center = NULL,
    normalization = c("det", "trace", "one"), maxiter = 1e4, eps = 1e-6)

classicShape(x, center = NULL,
    normalization = c("det", "trace", "one"), maxiter = 1e4, eps = 1e-6)
}
\arguments{
\item{x}{A numeric data matrix or data.frame without missing data.}

\item{alpha}{Tail index, a numeric value in the interval \code{[0, 1]}.
Determines the power function. For more information see 'Details'.}

\item{center}{An optional vector of the data's center. If \code{NULL} the center
will be estimated simultaneously with the shape.}

\item{normalization}{A string determining how the shape matrix is standardized.
The possible values are
\itemize{
\item{\code{'det'}}{such that the returned shape estimate has determinant 1.}
\item{\code{'trace'}}{such that the returned shape estimate has trace \code{ncol(x)}.}
\item{\code{'one'}}{such that the returned shape estimate's top left entry
(\code{S[1, 1]}) is 1.}
}}

\item{maxiter}{A positive integer, restricting the maximum number of iterations.}

\item{eps}{A numeric, specifying the tolerance level of when the iteration stops.}
}
\value{
A list with class 'shapeNA' containing the following components:
\item{S}{The estimated shape matrix.}
\item{scale}{The scale with which the shape matrix may be scaled to obtain
a scatter estimate. If \code{alpha} = 1, then this value is \code{NA}, as Tyler's
shape matrix has no natural scale.}
\item{mu}{The location parameter, either provided by the user or estimated.}
\item{alpha}{The tail index with which the Power M-estimator has been called.}
\item{naBlocks}{\code{NULL}, since \code{\link{powerShape}} operates only on complete data.}
\item{iterations}{Number of computed iterations before convergence.}
\item{call}{The matched call.}
}
\description{
Power M-estimators of shape and location were recently suggested in
Frahm et al. (2020). They have a tuning parameter \code{alpha} taking values in
\code{[0,1]}. The extreme case \code{alpha} = 1 corresponds to Tyler's shape
matrix and \code{alpha} = 0 to the classical covariance matrix. These special
cases have their own, more efficient functions \code{\link{tylerShape}} and
\code{\link{classicShape}}, respectively.
If the true location is known, it should be supplied as \code{center}, otherwise
it is estimated simultaneously with the shape.
}
\details{
These functions assume that the data were generated from an
elliptical distribution, for Tyler's estimate this can be relaxed to
generalized elliptical distributions.

For multivariate normally distributed data, \code{\link{classicShape}} is the maximum
likelihood estimator of location and scale. It is a special case of the
power M-estimator with tail index \code{alpha} = 0, which returns the empirical
covariance matrix and the empirical mean vector.

The function \code{\link{tylerShape}} maximizes the likelihood function after projecting
the observed data of each individual onto the unit hypersphere, in which case
we obtain an angular central Gaussian distribution. It is a special case of
the power M-estimator with tail index \code{alpha} = 1, which returns Tyler's
M-estimator of scatter and an affine equivariant multivariate median
according to Hettmansperger and Randles (2002).

The function \code{\link{powerShape}} requires an additional parameter, the so-called
tail index \code{alpha}. For heavy tailed data, the index should be chosen closer
to 1, whereas for light tailed data the index should be chosen closer to 0.
}
\examples{
    ## Generate example data
    S <- toeplitz(c(1, 0.1))
    x <- mvtnorm::rmvt(100, S)
    ## Compute some M-estimators
    res0 <- classicShape(x, center = c(0, 0))
    res1 <- powerShape(x, alpha = 0.67, normalization = 'one')
    res2 <- tylerShape(x, normalization = 'trace')
    ## Get location estimates
    res1$mu
    res2$mu
    ## Get shape estimates
    res0$S
    res1$S
    res2$S
    ## Print summary
    summary(res0)
}
\references{
Tyler, D.E. (1987). A Distribution-Free M-Estimator of Multivariate Scatter. The Annals of Statistics, 15, 234.251. \doi{10.1214/aos/1176350263}.

Frahm, G., Nordhausen, K., & Oja, H. (2020). M-estimation with incomplete and dependent multivariate data. Journal of Multivariate Analysis, 176, 104569. \doi{10.1016/j.jmva.2019.104569}.

Hettmansperger, T. P., & Randles, R. H. (2002). A practical affine equivariant multivariate median. Biometrika, 89(4), 851-860. \doi{10.1093/biomet/89.4.851}
}
\seealso{
\link{powerShapeNA}, \link{tylerShapeNA} and \link{classicShapeNA} for the
corresponding functions for data with missing values.
}
