% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summary.kernel_sievePH.R
\name{summary.kernel_sievePH}
\alias{summary.kernel_sievePH}
\alias{print.summary.kernel_sievePH}
\title{Summarizing Nonparametric Kernel-Smoothed Stratified Mark-Specific
Proportional Hazards Model Fits}
\usage{
\method{summary}{kernel_sievePH}(
  object,
  contrast = c("te", "hr", "loghr"),
  sieveAlternative = c("twoSided", "oneSided"),
  confLevel = 0.95,
  ...
)

\method{print}{summary.kernel_sievePH}(x, digits = 4, ...)
}
\arguments{
\item{object}{an object of class \code{kernel_sievePH}, a result of a call to
\code{\link{kernel_sievePH}} and \code{\link{kernel_sievePHaipw}}.}

\item{contrast}{a character string specifying the treatment effect parameter
of interest. The default value is \code{"te"} (treatment efficacy); other
options are \code{"hr"} (hazard ratio) and \code{"loghr"} (log hazard
ratio).}

\item{sieveAlternative}{a character string specifying the alternative
hypothesis for the sieve tests, which can be either \code{"twoSided"}
(default) or \code{"oneSided"}.}

\item{confLevel}{the confidence level (0.95 by default) of reported
confidence intervals}

\item{...}{further arguments passed to or from other methods}

\item{x}{an object of class \code{summary.kernel_sievePH}, usually a result of a call to \code{summary.kernel_sievePH}}

\item{digits}{the number of significant digits to use when printing (4 by default)}
}
\value{
An object of class \code{summary.kernel_sievePH}, which is a list
  with the following components:
\itemize{
\item \code{estBeta}: a data frame summarizing point estimates and standard
errors of the mark-specific coefficients for treatment.

\item \code{HRunity.2sided}: a data frame with test statistics (first row) and
corresponding p-values (second row) for testing \eqn{H_{10}: HR(v) = 1} vs.
\eqn{H_{1a}: HR(v) \neq 1} for any v \eqn{\in [a, b]} (general
alternative). \code{TSUP1} is based on an extension of the classic Kolmogorov-Smirnov
supremum-based test. \code{Tint1} is a generalization
of the integration-based Cramer-von Mises test.

\item \code{HRunity.1sided}: a data frame with test statistics (first row) and
corresponding p-values (second row) for testing \eqn{H_{10}: HR(v) = 1} vs.
\eqn{H_{1m}: HR(v) \leq 1} with strict inequality for some v \eqn{\in [a, b]}
(monotone alternative). \code{TSUP1m} is based on an extension of the classic
Kolmogorov-Smirnov supremum-based test. \code{Tint1m} is a generalization
of the integration-based Cramer-von Mises test.


\item \code{HRconstant.2sided}: a data frame with test statistics (first row) and
corresponding p-values (second row) for testing \eqn{H_{20}}: HR(v) does not depend on v
\eqn{\in [a, b]} vs. \eqn{H_{2a}}: HR depends on v \eqn{\in [a, b]}
(general alternative). \code{TSUP2} is based on an extension of the classic
Kolmogorov-Smirnov supremum-based test. \code{Tint2} is a generalization
of the integration-based Cramer-von Mises test.
This component is available if \code{sieveAlternative="twoSided"}.

\item \code{HRconstant.1sided}: a data frame with test statistics (first row) and
corresponding p-values (second row) for testing \eqn{H_{20}}: HR(v) does not depend on v
\eqn{\in [a, b]} vs. \eqn{H_{2m}}: HR increases as v increases \eqn{\in [a, b]}
(monotone alternative). \code{TSUP2m} is based on an extension of the classic
Kolmogorov-Smirnov supremum-based test. \code{Tint2m} is a generalization
of the integration-based Cramer-von Mises test.
This component is available if \code{sieveAlternative="oneSided"}.

\item \code{te}: a data frame summarizing point and interval estimates of the
mark-specific treatment efficacy on the grid of mark values defined by
\code{nvgrid} spanning from the minimum and maximum of the mark (available if \code{contrast="te"}).
The confidence level is specified by \code{confLevel}.
\item \code{hr}: a data frame summarizing point and interval estimates of the
mark-specific hazard ratio on the grid of mark values defined by
\code{nvgrid} spanning from the minimum and maximum of the mark (available if
\code{contrast="hr"}). The confidence level is specified by \code{confLevel}.
\item \code{loghr}: a data frame summarizing point and interval estimates of
the mark-specific log hazard ratio on the grid of mark values defined by
\code{nvgrid} spanning from the minimum and maximum of the mark (available if
\code{contrast="loghr"}). The confidence level is specified by
\code{confLevel}.
}
}
\description{
\code{summary} method for an object of class \code{kernel_sievePH}.
}
\details{
\code{print.summary.kernel_sievePH} prints a formatted summary of
  results. Inference about coefficients in the kernel-smoothed mark-specific
  proportional hazards model is tabulated. Additionally, a summary is
  generated
from the tests of two relevant null hypotheses: (1) \{\eqn{H_0: HR(v)=1} for
all \eqn{v}\}, and (2) \{\eqn{H_0: HR(v)=HR} for all \eqn{v}\}. For the tests
  of (2), \code{sieveAlternative} controls the choice of the alternative
  hypothesis.
}
\examples{
set.seed(20240410)
beta <- 2.1
gamma <- -1.3
n <- 200
tx <- rep(0:1, each = n / 2)
tm <- c(rexp(n / 2, 0.2), rexp(n / 2, 0.2 * exp(gamma)))
cens <- runif(n, 0, 15)
eventTime <- pmin(tm, cens, 3)
eventInd <- as.numeric(tm <= pmin(cens, 3))
alpha <- function(b){ log((1 - exp(-2)) * (b - 2) / (2 * (exp(b - 2) - 1))) }
mark0 <- log(1 - (1 - exp(-2)) * runif(n / 2)) / (-2)
mark1 <- log(1 + (beta - 2) * (1 - exp(-2)) * runif(n / 2) / (2 * exp(alpha(beta)))) /
  (beta - 2)
mark <- ifelse(eventInd == 1, c(mark0, mark1), NA)
# the true TE(v) curve underlying the data-generating mechanism is:
# TE(v) = 1 - exp{alpha(beta) + beta * v + gamma}

# a binary auxiliary covariate
A <- sapply(exp(-0.5 - 0.2 * mark) / (1 + exp(-0.5 - 0.2 * mark)),
            function(p){ ifelse(is.na(p), NA, rbinom(1, 1, p)) })
linPred <- 1 + 0.4 * tx - 0.2 * A
probs <- exp(linPred) / (1 + exp(linPred))
R <- rep(NA, n)
while (sum(R, na.rm = TRUE) < 10){
  R[eventInd == 1] <- sapply(probs[eventInd == 1],
                             function(p){ rbinom(1, 1, p) })
}
# a missing-at-random mark
mark[eventInd == 1] <- ifelse(R[eventInd == 1] == 1, mark[eventInd == 1], NA)

# AIPW estimation; auxiliary covariate is used (not required)
fit <- kernel_sievePHaipw(eventTime, eventInd, mark, tx, aux = A,
                          auxType = "binary", formulaMiss = ~ eventTime,
                          formulaAux = ~ eventTime + tx + mark,
                          tau = 3, tband = 0.5, hband = 0.3, nvgrid = 20,
                          nboot = 20)
sfit <- summary(fit)
# print the formatted summary
sfit
# treatment efficacy estimates on the grid
sfit$te

}
\references{
Gilbert, P. B. and Sun, Y. (2015). Inferences on relative failure rates in
stratified mark-specific proportional hazards models with missing marks, with
application to human immunodeficiency virus vaccine efficacy trials.
\emph{Journal of the Royal Statistical Society Series C: Applied Statistics},
64(1), 49-73.

Sun, Y. and Gilbert, P. B. (2012). Estimation of stratified mark‐specific
proportional hazards models with missing marks. \emph{Scandinavian Journal of
Statistics}, 39(1), 34-52.
}
\seealso{
\code{\link{kernel_sievePH}}
}
