% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bartlett.R
\name{bartlettFactor}
\alias{bartlettFactor}
\title{Bartlett correction factor for empirical likelihood with estimating equations}
\usage{
bartlettFactor(x, centre = TRUE, bias.adj = TRUE)
}
\arguments{
\item{x}{Numeric vector or matrix of estimating functions. If a matrix,
rows are observations and columns are the components of \eqn{g}.}

\item{centre}{Logical. If `TRUE` (default), centre each column of `x` by its
sample mean before computing the correction (this corresponds to plugging in
a consistent \eqn{\hat\theta} so that
\eqn{n^{-1}\sum g_i(\hat\theta)\approx 0}{1/n sum_i g_i(^theta) ~ 0}).}

\item{bias.adj}{Logical. If `TRUE` (default), use the bias-reduced
moment estimators. When \eqn{n \le 4}, the adjustment is disabled automatically.}
}
\value{
Numeric scalar: the estimated Bartlett correction factor \eqn{b}.
  For multivariate inputs, the value has an attribute \code{"components"} equal to
  \code{c(b1, b2)} where \code{b = b_1 - b_2}. If \code{bias.adj = TRUE}, attributes
  \code{"unadjusted"} and \code{"unadjusted.components"} store the corresponding
  unadjusted estimates.
}
\description{
Compute the Bartlett correction factor \eqn{b} for empirical likelihood based on
the moment conditions \eqn{E\{g(X;\theta_0)\}=0}. The function implements the rotation
in \insertCite{liu2010adjusted}{smoothemplik} and evaluates \eqn{b} either from raw moments (unadjusted) or from
the bias-reduced moment estimators recommended in their paper.
}
\details{
Let \eqn{V(\theta) = \mathrm{Var}\{g(X, \theta)\}}{V(theta) = Var(g(X, theta))},
and let \eqn{P}{P} be the orthogonal matrix of eigenvectors of \eqn{V(\hat\theta)}{V(^theta)}.
Define the rotated variables \eqn{Y_i = g_i(\hat\theta) P} (observations in rows), and write
\eqn{\alpha^{rs\cdots t} = E(Y^r Y^s \cdots Y^t)}{ars..t = E(Y^r Y^s ... Y^t)} with
\eqn{\alpha^{rr}=E(Y_r^2)}{arr = E(Y[, r]^2)}.

The Bartlett factor (Theorem 1 of \insertCite{liu2010adjusted}{smoothemplik}) can be written compactly as
\deqn{
b = \frac{1}{q}\Bigl\{
  \frac{1}{2} \sum_{r,s} \frac{\alpha^{rrss}}{\alpha^{rr} \alpha^{ss}}
  -
  \frac{1}{3} \sum_{r,s,t} \frac{(\alpha^{rst})^2}{\alpha^{rr} \alpha^{ss} \alpha^{tt}}
\Bigr\},
}
where \eqn{q} is the dimension of \eqn{g}. The first double sum is over all
pairs \code{(r, s)}, and the triple sum is over all triples \code{(r, s, t)}.

For adjusted-EL applications, the implementation also uses the equivalent
decomposition \code{b = b_1 - b_2}.

When \code{bias.adj = TRUE}, all moments are replaced by the
bias-reduced estimators given in Eq. (10) and the table beneath it in
\insertCite{liu2010adjusted}{smoothemplik}.
}
\examples{
set.seed(1)

# One-dimensional: Bartlett factor for the mean
x <- rchisq(50, df = 4)
bartlettFactor(x)  # Bias-adjusted
bartlettFactor(x, bias.adj=FALSE)

# Multi-variate g(X; theta): columns are components of g
n <- 100
g <- cbind(rchisq(n, 4)-4, rchisq(n, 3)-3, rchisq(n, 6)-6, rnorm(n))
bartlettFactor(g)  # Bias-adjusted, centred
bartlettFactor(g, centre = FALSE)  # The true average was used in g
}
\references{
\insertAllCited{}
}
