% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DSC_evoStream.R
\name{DSC_evoStream}
\alias{DSC_evoStream}
\title{evoStream - Evolutionary Stream Clustering}
\usage{
DSC_evoStream(
  formula = NULL,
  r,
  lambda = 0.001,
  tgap = 100,
  k = 2,
  crossoverRate = 0.8,
  mutationRate = 0.001,
  populationSize = 100,
  initializeAfter = 2 * k,
  incrementalGenerations = 1,
  reclusterGenerations = 1000
)
}
\arguments{
\item{formula}{\code{NULL} to use all features in the stream or a model \link{formula} of the form \code{~ X1 + X2}
to specify the features used for clustering. Only \code{.}, \code{+} and \code{-} are currently
supported in the formula.}

\item{r}{radius threshold for micro-cluster assignment}

\item{lambda}{decay rate}

\item{tgap}{time-interval between outlier detection and clean-up}

\item{k}{number of macro-clusters}

\item{crossoverRate}{cross-over rate for the evolutionary algorithm}

\item{mutationRate}{mutation rate for the evolutionary algorithm}

\item{populationSize}{number of solutions that the evolutionary algorithm
maintains}

\item{initializeAfter}{number of micro-cluster required for the
initialization of the evolutionary algorithm.}

\item{incrementalGenerations}{number of EA generations performed after each
observation}

\item{reclusterGenerations}{number of EA generations performed during
reclustering}
}
\description{
Micro Clusterer with reclustering.
Stream clustering algorithm based on evolutionary optimization.
}
\details{
The online component uses a simplified version of \link{DBSTREAM} to
generate micro-clusters. The micro-clusters are then incrementally
reclustered using an evolutionary algorithm. Evolutionary algorithms create
slight variations by combining and randomly modifying existing solutions. By
iteratively selecting better solutions, an evolutionary pressure is created
which improves the clustering over time. Since the evolutionary algorithm is
incremental, it is possible to apply it between observations, e.g. in the
idle time of the stream. Whenever there is idle time, we can call the
\code{\link[=recluster]{recluster()}} function of the reference class to improve the
macro-clusters (see example). The evolutionary algorithm can also be applied
as a traditional reclustering step, or a combination of both. In addition,
this implementation also allows to evaluate a fixed number of generations
after each observation.
}
\examples{
stream <- DSD_Gaussians(k = 3, d = 2) \%>\% DSD_Memory(n = 500)

## init evoStream
evoStream <- DSC_evoStream(r = 0.05, k = 3,
  incrementalGenerations = 1, reclusterGenerations = 500)

## insert observations
update(evoStream, stream, n = 500)

## micro clusters
get_centers(evoStream, type = "micro")

## micro weights
get_weights(evoStream, type = "micro")

## macro clusters
get_centers(evoStream, type = "macro")

## macro weights
get_weights(evoStream, type = "macro")

## plot result
reset_stream(stream)
plot(evoStream, stream)

## if we have time, then we can evaluate additional generations.
## This can be called at any time, also between observations.
## by default, 1 generation is evaluated after each observation and
## 1000 generations during reclustering but we set it here to 500
evoStream$RObj$recluster(500)

## plot improved result
reset_stream(stream)
plot(evoStream, stream)

## get assignment of micro to macro clusters
microToMacro(evoStream)
}
\references{
Carnein M. and Trautmann H. (2018), "evoStream - Evolutionary
Stream Clustering Utilizing Idle Times", Big Data Research.
}
\seealso{
Other DSC_Micro: 
\code{\link{DSC_BICO}()},
\code{\link{DSC_BIRCH}()},
\code{\link{DSC_DBSTREAM}()},
\code{\link{DSC_DStream}()},
\code{\link{DSC_Micro}()},
\code{\link{DSC_Sample}()},
\code{\link{DSC_Window}()}

Other DSC_TwoStage: 
\code{\link{DSC_DBSTREAM}()},
\code{\link{DSC_DStream}()},
\code{\link{DSC_TwoStage}()}
}
\author{
Matthias Carnein \email{Matthias.Carnein@uni-muenster.de}
}
\concept{DSC_Micro}
\concept{DSC_TwoStage}
