% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cut.R
\name{cut}
\alias{cut}
\alias{cut.tind}
\title{Group Time Indices into Periods / Convert to a Factor}
\usage{
\method{cut}{tind}(x, breaks, labels = TRUE, right = FALSE, ...)
}
\arguments{
\item{x}{an object of \code{tind} class.}

\item{breaks}{a numeric value or a character string determining intervals,
or an object of \code{tind} class with cut points, see Details.}

\item{labels}{a logical value controlling the return type, which
can be a factor (if \code{TRUE}, the default), integer vector,
or a 2-element list.}

\item{right}{a logical value determing whether indices should be matched
to the closest left cut point or to the closest right cut point,
see Details.}

\item{...}{(ignored) further arguments passed to or from other methods.}
}
\value{
A factor if \code{labels} is \code{TRUE}, an integer vector if \code{FALSE},
and a 2-element list if \code{NA}, see Details.
}
\description{
\code{cut} method for objects of \code{tind} class allows to map / group
time indices into periods. The periods can be determined based on indices
provided by the user or by (multiples of) units of time.
}
\details{
\code{breaks} argument controls how indices are grouped. It can be a number
or a character string determining resolution (or an object of \code{tdiff} class).
Alternatively, \code{breaks} can be an object of \code{tind} class with cut points.

When \code{breaks} determines resolution, only selected multiples of units
are allowed, similarly to \code{floor_t} function. Documentation of admissible
units and multiples can be found in Details section of \code{\link{resolution_t}}
method documentation. If selected resolution corresponds to an index of different
type (for example grouping dates to 2-month periods), conversion takes place.

This method differs from \code{\link[base]{cut.POSIXt}} and \code{\link[base]{cut.Date}}
in two aspects. Firstly, the periods are selected differently, they are always
aligned to resolution, see Examples. Secondly, as it does not rely
on \code{seq} but rounding of indices, the levels may be discontinuous. If users want
to replicate behaviour of \code{cut} from \pkg{base}, they should provide \code{tind}
constructed via \code{\link{seq.tind}} method as \code{breaks} argument.

When \code{breaks} is a \code{tind} object, it is expected to be sorted
without \code{NA}s. By default, indices in \code{x}
are matched to the closest index to the left (largest index that is not greater
than the argument). If \code{right} is set to \code{TRUE}, indices are matched
to the closest index to the right (smallest index that is not smaller than
the argument). \code{right} cannot be set to \code{TRUE} if \code{breaks}
is not a \code{tind}. It is acceptable that \code{breaks} is of lower resolution
than \code{x} provided that \code{x} is convertible to it. In such situations,
\code{right} cannot be set to \code{TRUE}.

By default, \code{cut.tind} returns a factor with levels created using
\code{as.character} method. If \code{labels} argument is set
to \code{FALSE}, only the integer vector (of the same length as argument)
of mappings to intervals is returned (as in \pkg{base} method).
If set to \code{NA}, a 2-element list is returned, with integer vector
of mappings as the first element and time indices determining intervals
(grouping, levels) as the second.
\code{labels} can only take \code{TRUE/FALSE/NA} values.
}
\examples{
# basic use
(d <- seq.tind("2023-09-14", "2023-12-16"))
cut(d, "15d")
cut(d, "m")
cut(d, "2m")
# tind given as breaks
cut(d, as.date(c("2023-09-01", "2023-11-16", "2023-12-16")))
cut(d, seq.tind("2023-01", "2023-12"))
# random order with NAs
(d <- sample(c(d, NA)))
cut(d, "15d")
cut(d, "m")
cut(d, "2m")
# different behaviour of cut for tind and Date (alignment to 2 month resolution,
# which means Jan, Mar, May, Jul, Sep, Nov)
(d <- seq.tind("2023-12-16", "2024-03-01"))
cut(d, "2 months")
cut(as.Date(d), "2 months")
# replicate behviour of cut.Date by providing sequence of months
cut(d, seq.tind("2023-12", "2024-03", by = "2m"))
# same
cut(d, seq.tind(as.month(min(d)), as.month(max(d)), by = "2m"))
# check
all.equal(cut(as.Date(d), "2 months", labels = FALSE),
          cut(d, seq.tind("2023-12", "2024-03", by = "2m"), labels = FALSE))

}
\seealso{
\link{rounding} and \code{\link{resolution_t}} for description
of admissible units and multiples that can be used for \code{breaks} argument.
\code{\link{match_t}} for matching time indices to other indices and time
intervals.
}
