% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HACLasso.r
\name{rlassoHAC}
\alias{rlassoHAC}
\title{\code{rlassoHAC} performs Lasso estimation under heteroscedastic and autocorrelated non-Gaussian disturbances.}
\source{
Victor Chernozhukov, Chris Hansen, Martin Spindler (2016). hdm: High-Dimensional Metrics,
R Journal, 8(2), 185-199. URL https://journal.r-project.org/archive/2016/RJ-2016-040/index.html.
}
\usage{
rlassoHAC(
  x,
  y,
  kernel = "Bartlett",
  bands = 10,
  bns = 10,
  lns = NULL,
  nboot = 5000,
  post = TRUE,
  intercept = TRUE,
  model = TRUE,
  X.dependent.lambda = FALSE,
  c = 2,
  gamma = NULL,
  numIter = 15,
  tol = 10^-5,
  threshold = NULL,
  ...
)
}
\arguments{
\item{x}{Regressors (vector, matrix or object can be coerced to matrix).}

\item{y}{Dependent variable (vector, matrix or object can be coerced to matrix).}

\item{kernel}{Kernel function, choose between "Truncated", "Bartlett" (by default), "Parzen", 
"Tukey-Hanning", "Quadratic Spectral".}

\item{bands}{Bandwidth parameter with default bands=10.}

\item{bns}{Block length with default bns=10.}

\item{lns}{Number of blocks with default lns = floor(T/bns).}

\item{nboot}{Number of bootstrap iterations with default nboot=5000.}

\item{post}{Logical. If TRUE (default), post-Lasso estimation is conducted, i.e. a refit of the model with the selected variables.}

\item{intercept}{Logical. If TRUE, intercept is included which is not penalized.}

\item{model}{Logical. If TRUE (default), model matrix is returned.}

\item{X.dependent.lambda}{Logical, TRUE, if the penalization parameter depends on the design
of the matrix x. FALSE (default), if independent of the design matrix.}

\item{c}{Constant for the penalty, default value is 2.}

\item{gamma}{Constant for the penalty, default gamma=0.1/log(T) with T=data length.}

\item{numIter}{Number of iterations for the algorithm for the estimation of the variance and
data-driven penalty, ie. loadings.}

\item{tol}{Constant tolerance for improvement of the estimated variances.}

\item{threshold}{Constant applied to the final estimated lasso coefficients. Absolute values 
below the threshold are set to zero.}

\item{...}{further parameters}
}
\value{
rlassoHAC returns an object of class "rlasso". An object of class "rlasso" is a list containing at least the
following components:
\item{coefficients}{Parameter estimates.}
\item{beta}{Parameter estimates (named vector of coefficients without intercept). }
\item{intercept}{Value of the intercept. }
\item{index}{Index of selected variables (logical vector). }
\item{lambda}{Data-driven penalty term for each variable, product of lambda0 (the penalization parameter) and the loadings. }
\item{lambda0}{Penalty term. }
\item{loadings}{Penalty loadings, vector of lenght p (no. of regressors). }
\item{residuals}{Residuals, response minus fitted values. }
\item{sigma}{Root of the variance of the residuals. }
\item{iter}{Number of iterations. }
\item{call}{Function call. }
\item{options}{Options. }
\item{model}{Model matrix (if model = TRUE in function call). }
}
\description{
\code{rlassoHAC} performs Lasso estimation under heteroscedastic and autocorrelated non-Gaussian disturbances.
}
\examples{
 
\donttest{
set.seed(1)
T = 100 #sample size
p = 20 # number of variables
b = 5 # number of variables with non-zero coefficients
beta0 = c(rep(10,b), rep(0,p-b))
rho = 0.1 #AR parameter
Cov = matrix(0,p,p)
for(i in 1:p){
  for(j in 1:p){
     Cov[i,j] = 0.5^(abs(i-j))
  }
} 
C <- chol(Cov)
X <- matrix(rnorm(T*p),T,p)\%*\%C
eps <- arima.sim(list(ar=rho), n = T+100)
eps <- eps[101:(T+100)] 
Y = X\%*\%beta0 + eps
reg.lasso.hac1 <- rlassoHAC(X, Y,"Bartlett") #lambda is chosen independent of regressor 
                                             #matrix X by default.

bn = 10 # block length
bwNeweyWest = 0.75*(T^(1/3))
reg.lasso.hac2 <- rlassoHAC(X, Y,"Bartlett", bands=bwNeweyWest, bns=bn, nboot=5000,
                            X.dependent.lambda = TRUE, c=2.7) 
}
}
