#' Estimating under-five mortality using Coale-Demeny life table models
#'
#' @description
#'
#' `r lifecycle::badge("stable")`
#'
#' `u5mr_trussell()` uses the Trussell version of the BRASS method
#' and calculates under-five mortality estimates.
#'
#' @param data processed data
#' @param women total number of women
#' @param child_born children ever born
#' @param child_dead children dead
#' @param agegrp age grouping
#' @param model Coale-Demeny life table model:
#' `north`, `south`, `east`, and `west`
#' @param sex indicates sex-specific estimates: `both`, `male`, and `female`
#' @param svy_year end of the survey
#'
#'
#' @details
#' T. J. Trussell developed the Trussell version of the Brass method to estimate
#' child mortality using summary birth history (United Nations,  1983b, Chapter III).
#' It is based on the Coale-Demeny life table models of either North, South, East, or West.
#'
#' \strong{Computational Procedure}
#'
#' Step 1. Preparing the dataset
#'
#' The function needs four variables from the input data:
#'
#' a) `agegrp`: age groups representing `15-19`, `20-24`, `25-29`, `30-34`,
#' `35-39`, `40-44`, and `45-49`.
#'
#' b) `women`: the total number of women in the age group irrespective of their marital
#' or reporting status
#'
#' c) `child_born`: the total number of children ever borne by these women
#'
#' d) `child_dead`: the number of children dead reported by these women
#'
#' Step 1.1. Calculation of average parity per woman `P(i)`
#'
#' \deqn{P(i) = CEB(i) / FP(i)}
#'
#' - `CEB(i)` is the total number of children ever borne by these women
#'
#' - `FP(i)` is the total number of women in the
#' age group irrespective of their marital or reporting status.
#'
#' Step 1.2. Calculation of the proportions dead among children ever borne `D(i)`
#'
#' \deqn{D(i) = CD(i) / CEB(i)}
#'
#' - `CD(i)` is the number of children dead for women of age group i
#'
#'
#'
#' Step 2. Calculating the multipliers `k(i)` and probabilities of dying by age x `q(x)`
#'
#' \deqn{k(i) = a(i) + b(i) P(1)/P(2) + c(i) P(2)/P(3)}
#'
#' where `a(i)`, `b(i)`, and `c(i)` are coefficients estimated by regression analysis of
#' simulated model cases, and `P(1)/P(2)` and `P(2)/P(3)` are parity ratios.
#'
#' \deqn{q(x) = k(i) x D(i)}
#'
#'
#' Step 3. Calculating the reference dates for `q(x)` and interpolating `q5`
#'
#' Under conditions of steady mortality change over time, a reference time `t(i)`
#' can be estimated for each `q(x)`.
#'
#' \deqn{t(i) = a(i) + b(i) P(1)/P(2) + c(i) P(2)/P(3)}
#'
#' Actual dates can be obtained by subtracting `t(i)` from the reference date of
#' the survey or census, expressed in decimal terms.
#'
#'
#' Step 4. Interpolating between `q(x)` and model life table
#'
#' A common index, in this case, under-five mortality `q(5)` can be obtained by
#' conversions of corresponding `q(x)` values to the specified family of
#' the Coale-Demeny life table models. In a given life table family and sex,
#' the first step is to identify the mortality levels with q(x) values that
#' enclose the estimated value \eqn{q^e(x)}.
#'
#' \deqn{q^j(x) > q^e(x) > q^j+1(x)}
#'
#' where \eqn{q^j(x)} and \eqn{q^j+1(x)} are the model values of `q(x)` for
#' levels `j` and `j+1`, and \eqn{q^e(x)} is the estimated value.
#'
#' The desired common index \eqn{q^c(5)}, or `q5` is given by
#'
#'
#' \deqn{q^c(5) = (1.0 - h) x q^j(5) + h x q^e(5)}
#'
#' where h is the interpolation factor calculated in the following way:
#'
#' \deqn{h = q^e(x) - q^j(x) / q^j+1(x) - q^j(x)}
#'
#'
#' Step 5. Finalizing the calculation
#'
#' The final output is combined into a `data.frame`, which contains original dataset
#' as well as statistics produced during the computational procedure.
#'
#' @references
#'
#' 1. United Nations (1990) "Step-by-step guide to the estimation of the child mortality"
#' <https://www.un.org/en/development/desa/population/publications/pdf/mortality/stepguide_childmort.pdf>
#' 2. United Nations (1983) "Manual X indirect techniques for demographic estimation"
#' <https://www.un.org/en/development/desa/population/publications/pdf/mortality/stepguide_childmort.pdf>
#'
#' @return `data.frame`
#'
#' * `agegrp`      - five-year age groups
#' * `women` - total number of women
#' * `child_born`  - total number of children ever born
#' * `child_dead`  - number of children dead
#' * `pi` - average parity
#' * `di` - proportion of dead among children ever born
#' * `ki` - multipliers
#' * `qx` - probabilities of dying at age x
#' * `ti` - time between survey year and interpolated year
#' * `year` - reference year
#' * `h`  - interpolation factor
#' * `q5` - under-five mortality
#'
#' @examples
#'
#'## Using Bangladesh survey data to estimate child mortality
#'data("bangladesh")
#'bang_both <- u5mr_trussell(bangladesh, sex = "both", model = "south", svy_year = 1974.3)
#'bang_male <- u5mr_trussell(bangladesh, child_born = "male_born",
#'                  child_dead = "male_dead", sex = "male",
#'                  model = "south", svy_year = 1974.3)
#'bang_female <- u5mr_trussell(bangladesh, child_born = "female_born",
#'                  child_dead = "female_dead", sex = "female",
#'                  model = "south", svy_year = 1974.3)
#'
#'## plotting all data points
#'with(bang_both,
#'     plot(year, q5, type = "b", pch = 19,
#'          ylim = c(0, .45),
#'          col = "black", xlab = "Reference date", ylab = "u5MR",
#'          main = paste0("Under-five mortality, q(5) in Bangladesh, estimated\n",
#'                        "using model South and the Trussell version of the Brass method")))
#'with(bang_both, text(year, q5, agegrp, cex=0.65, pos=3,col="purple"))
#'with(bang_male,
#'     lines(year, q5, pch = 18, col = "red", type = "b", lty = 2))
#'with(bang_female,
#'     lines(year, q5, pch = 18, col = "blue", type = "b", lty = 3))
#'legend("bottomright", legend=c("Both sexes", "Male", "Female"),
#'       col = c("Black", "red", "blue"), lty = 1:3, cex=0.8)
#'
#'
#'## Using panama survey data to estimate child mortality
#'data("panama")
#'pnm_both <- u5mr_trussell(panama, sex = "both", model = "west", svy_year = 1976.5)
#'pnm_male <- u5mr_trussell(panama, child_born = "male_born",
#'                 child_dead = "male_dead", sex = "male",
#'                 model = "west", svy_year = 1976.5)
#'pnm_female <- u5mr_trussell(panama, child_born = "female_born",
#'                 child_dead = "female_dead", sex = "female",
#'                 model = "west", svy_year = 1976.5)
#'
#'## plotting all data points
#'with(pnm_both,
#'     plot(year, q5, type = "b", pch = 19,
#'         ylim = c(0, .2), col = "black", xlab = "Reference date", ylab = "u5MR",
#'          main = paste0("Under-five mortality, q(5) in Panama, estimated\n",
#'                        "using model West and the Trussell version of the Brass method")))
#'with(pnm_both, text(year, q5, agegrp, cex=0.65, pos=3,col="purple"))
#'with(pnm_male,
#'     lines(year, q5, pch = 18, col = "red", type = "b", lty = 2))
#'with(pnm_female,
#'     lines(year, q5, pch = 18, col = "blue", type = "b", lty = 3))
#'legend("bottomleft", legend=c("Both sexes", "Male", "Female"),
#'       col = c("Black", "red", "blue"), lty = 1:3, cex=0.8)
#'
#' @export
u5mr_trussell <- function(data, women = "women",
                          child_born = "child_born", child_dead = "child_dead",
                          agegrp = "agegrp", model = "west", svy_year = 1976.5, sex) {

  # message("  1. preparing data ... ")

  agegrp <- data[[agegrp]]
  women <- data[[women]]
  child_born <- data[[child_born]]
  child_dead <- data[[child_dead]]

  pi <- child_born / women
  di <- child_dead / child_born
  p1 <- pi[1]
  p2 <- pi[2]
  p3 <- pi[3]

  # message("  2. calculating multipliers k(i) and q(x) ... ")

  ## get Coefficients for estimation of child mortality multipliers k(i)
  ##    TRUSSELL variant TABLE 47 PAGE 77 UN Manual X
  coeff_trussell_ki <- coeff_trussell_ki()
  t47 <- coeff_trussell_ki[coeff_trussell_ki$model == model, -1]
  ki <- t47$ai + (t47$bi * p1/p2) + (t47$ci * p2/p3)
  qx <- ki * di


  # message("  3. calculating time reference t(i) and reference date ... ")

  ## get Coefficients for estimation of the reference period, t(x)
  ##    TRUSSELL variant TABLE 47 PAGE 77 UN Manual X
  coeff_trussell_ti <- coeff_trussell_ti()
  t48 <- coeff_trussell_ti[coeff_trussell_ti$model == model, -1]
  ti <- t48$ai + (t48$bi * p1/p2) + (t48$ci * p2/p3)
  year <- round(svy_year - ti, 1)

  # message("  4. Interpolation between qx and Coale-Demeny model for q(5) ... ")
  ## interpolation between qx and model for common index q5\
  coale_demeny_ltm <- coale_demeny_ltm()
  inter <- coale_demeny_ltm[[paste0(model, "_", sex)]]

  age_index <- paste0("q", c(1, 2, 3, 5, 10, 15, 20))

  h <- vector("numeric", length = 7L)
  q5 <- sapply(1:7, function(x) {
    q <- qx[x]
    c <- inter[[age_index[x]]]
    qj <- c[ c < q]
    qj1 <- c[ q < c]
    if (length(qj) == 0) qj <- 0

    h[x] <<- (q - qj[1]) / (qj1[length(qj1)] - qj[1])

    qj5 <- inter$q5[c == qj[1]]
    qj51 <- inter$q5[c == qj1[length(qj1)]]
    (1 - h[x]) * qj5 + (h[x] * qj51)
  })

  # message("  5. Finalizing the calculation ... ")
  data.frame(agegrp, women, child_born, child_dead,
             pi, di, ki, qx, ti, year, h, q5)
}




# HELPERS -----------------------------------------------------------------

coeff_trussell_ki <- function() {
  data.frame(
    model = rep(c("north", "south", "east", "west"), each = 7),
    agegrp = c("15-19", "20-24", "25-29", "30-34", "35-39", "40-44", "45-49"),
    ai = c(1.1119, 1.239, 1.1884, 1.2046, 1.2586, 1.224, 1.1772, 1.0819, 1.2846, 1.2223, 1.1905,
           1.1911, 1.1564, 1.1307, 1.1461, 1.2231, 1.1593, 1.1404, 1.154, 1.1336, 1.1201, 1.1415,
           1.2563, 1.1851, 1.172, 1.1865, 1.1746, 1.1639),
    bi = c(-2.9287, -0.6865, 0.0421, 0.3037, 0.4236, 0.4222, 0.3486, -3.0005, -0.6181, 0.0851,
           0.2631, 0.3152, 0.3017, 0.2596, -2.2536, -0.4301, 0.0581, 0.1991, 0.2511, 0.2556, 0.2362,
           -2.707, -0.5381, 0.0633, 0.2341, 0.308, 0.3314, 0.319),
    ci = c(0.8507, -0.2745, -0.5156, -0.5656, -0.5898, -0.5456, -0.4624, 0.8689, -0.3024, -0.4704,
           -0.4487, -0.4291, -0.3958, -0.3538, 0.6259, -0.2245, -0.3479, -0.3487, -0.3506, -0.3428,
           -0.3268, 0.7663, -0.2637, -0.4177, -0.4272, -0.4452, -0.4537, -0.4435)
  )
}


coeff_trussell_ti <- function() {
  data.frame(
    model = rep(c("north", "south", "east", "west"), each = 7),
    agegrp = c("15-19", "20-24", "25-29", "30-34", "35-39", "40-44", "45-49"),
    ai = c(1.0921, 1.3207, 1.5996, 2.0779, 2.7705, 4.152, 6.965, 1.09,
           1.3079, 1.5173, 1.9399, 2.6157, 4.0794, 7.1796, 1.0959, 1.2921,
           1.5021, 1.9347, 2.6197, 4.1317, 7.3657, 1.097, 1.3062, 1.5305,
           1.9991, 2.7632, 4.3468, 7.5242),
    bi = c(5.4732, 5.3751, 2.6268, -1.7908, -7.3403, -12.2448, -13.916,
           5.4443, 5.5568, 2.6755, -2.2739, -8.4819, -13.8308, -15.388,
           5.5864, 5.5897, 2.4692, -2.6419, -8.9693, -14.355, -15.8083,
           5.5628, 5.5677, 2.5528, -2.4261, -8.4065, -13.2436, -14.2013),
    ci = c(-1.9672, 0.2133, 4.3701, 9.4126, 14.9352, 19.2349, 19.9542,
           -1.9721, 0.2021, 4.7471, 10.3876, 16.5153, 21.1866, 21.7892,
           -1.9949, 0.3631, 5.0927, 10.8533, 17.0981, 21.8247, 22.3005,
           -1.9956, 0.2962, 4.8962, 10.4282, 16.1787, 20.199, 20.0162)
  )
}


coale_demeny_ltm <- function() {
  ## Step-by-Step Guideto the Estimation of Child Mortality
  ## United Nations, 1990

  ## COALE-DEMENY MORTALITY MODELS

  ## TABLE A.I.I. NORTH MODEL VALUES FOR MALE PROBABILITIES OF DYING. q(x)
  ## page 61
  north_male <- data.frame(
    e0 = c(17.6, 19.9, 22.3, 24.7, 27.2, 29.6, 32, 34.5, 36.9, 39.3, 41.8, 44.3, 46.7, 49.1, 51.5, 53.9, 56.3, 58.8, 61.3, 63.9, 66.4, 68.9, 71.6, 74.4, 77.3),
    q1 = c(0.37117, 0.33923, 0.31056, 0.28459, 0.26089, 0.23913, 0.21904, 0.20041, 0.18306, 0.16686, 0.15167, 0.13744, 0.12411, 0.11228, 0.10074, 0.08955, 0.07874, 0.06833, 0.05836, 0.04885, 0.03981, 0.0313, 0.02353, 0.01606, 0.01056),
    q2 = c(0.45216, 0.41659, 0.38401, 0.35397, 0.32613, 0.30021, 0.27596, 0.25322, 0.23181, 0.21161, 0.19251, 0.17444, 0.15672, 0.1408, 0.12544, 0.10995, 0.09527, 0.08145, 0.06844, 0.05622, 0.04476, 0.03405, 0.02516, 0.0169, 0.01096),
    q3 = c(0.50142, 0.46363, 0.42867, 0.39617, 0.36581, 0.33735, 0.31058, 0.28533, 0.26146, 0.23883, 0.21734, 0.19694, 0.17656, 0.15814, 0.14046, 0.12293, 0.10632, 0.09063, 0.07584, 0.06189, 0.04874, 0.03635, 0.02659, 0.01766, 0.01134),
    q5 = c(0.56587, 0.52518, 0.48711, 0.45138, 0.41773, 0.38595, 0.35588, 0.32735, 0.30024, 0.27443, 0.24983, 0.22639, 0.20251, 0.18084, 0.1601, 0.14031, 0.12145, 0.10348, 0.08638, 0.07011, 0.05462, 0.03979, 0.02876, 0.01883, 0.01193),
    q10 = c(0.62135, 0.5804, 0.54145, 0.50437, 0.46899, 0.4352, 0.40287, 0.37191, 0.34222, 0.31373, 0.28635, 0.26008, 0.23311, 0.20839, 0.18463, 0.16186, 0.14006, 0.11924, 0.09935, 0.08037, 0.06224, 0.0449, 0.03219, 0.02084, 0.01303),
    q15 = c(0.64586, 0.60516, 0.56616, 0.52877, 0.49289, 0.45842, 0.42527, 0.39338, 0.36266, 0.33306, 0.30452, 0.27703, 0.24884, 0.22285, 0.19778, 0.17366, 0.15051, 0.12831, 0.10706, 0.08673, 0.06727, 0.04863, 0.03487, 0.02255, 0.01408),
    q20 = c(0.66959, 0.62944, 0.5907, 0.55331, 0.51722, 0.48238, 0.44871, 0.41617, 0.3847, 0.35426, 0.32481, 0.29635, 0.26736, 0.24042, 0.21429, 0.18903, 0.16466, 0.1412, 0.11865, 0.097, 0.07621, 0.05622, 0.04062, 0.02674, 0.01702)
  )




  ## TABLEA.I.2. SOUTH MODEL VALUES FOR MALE PROBABILITIES OF DYING, q(X)
  ## page 62
  south_male <- data.frame(
    e0 = c(19.9, 22.3, 24.7, 27, 29.3, 31.6, 33.9, 36.2, 38.5, 40.6, 42.9, 45.1, 47.4, 49.6, 51.9, 54.1, 56.3, 58.6, 61.2, 63.7, 66.1, 68.5, 71, 73.6, 76),
    q1 = c(0.33555, 0.31122, 0.28925, 0.26924, 0.25089, 0.23396, 0.21828, 0.20368, 0.19004, 0.17752, 0.16642, 0.15559, 0.14502, 0.13475, 0.12478, 0.11513, 0.10581, 0.09676, 0.08618, 0.0761, 0.06605, 0.05605, 0.04617, 0.03648, 0.02822),
    q2 = c(0.46045, 0.42918, 0.40024, 0.3733, 0.34813, 0.32449, 0.30224, 0.28122, 0.26132, 0.24318, 0.22532, 0.20812, 0.19155, 0.17558, 0.16022, 0.14545, 0.13125, 0.11733, 0.10112, 0.08735, 0.0742, 0.06167, 0.0498, 0.03863, 0.02943),
    q3 = c(0.51806, 0.48359, 0.45143, 0.42131, 0.39298, 0.36625, 0.34097, 0.31699, 0.2942, 0.27346, 0.25249, 0.23235, 0.213, 0.19442, 0.17657, 0.15944, 0.14299, 0.12693, 0.10835, 0.093, 0.07845, 0.06472, 0.05184, 0.03988, 0.03016),
    q5 = c(0.56599, 0.52885, 0.49402, 0.46124, 0.43029, 0.40099, 0.37319, 0.34674, 0.32155, 0.29866, 0.2751, 0.25251, 0.23086, 0.21099, 0.19017, 0.17107, 0.15275, 0.13502, 0.11472, 0.09817, 0.08248, 0.0677, 0.05391, 0.04118, 0.03094),
    q10 = c(0.60074, 0.56317, 0.52756, 0.49374, 0.46154, 0.43083, 0.40148, 0.37339, 0.34647, 0.32152, 0.296, 0.27145, 0.24786, 0.22518, 0.20338, 0.18242, 0.16228, 0.14275, 0.12079, 0.10291, 0.08604, 0.07024, 0.05561, 0.04223, 0.03155),
    q15 = c(0.616, 0.57843, 0.54267, 0.50857, 0.47599, 0.4448, 0.41491, 0.38622, 0.35865, 0.33299, 0.30675, 0.28147, 0.25711, 0.23364, 0.21104, 0.18928, 0.16833, 0.14798, 0.1252, 0.10655, 0.08896, 0.07251, 0.05727, 0.04336, 0.03229),
    q20 = c(0.63801, 0.60052, 0.56461, 0.53016, 0.49707, 0.46525, 0.4346, 0.40507, 0.37659, 0.35007, 0.32276, 0.29637, 0.27086, 0.24261, 0.22242, 0.19946, 0.1773, 0.15574, 0.13164, 0.11186, 0.0932, 0.07575, 0.05962, 0.04494, 0.03332)
  )




  ## TABLE A,I.3. EAST MODEL VALUES FOR MALE PROBABILITIES OF DYING, q(X)
  ## page 62
  east_male <- data.frame(
    e0 = c(17.4, 19.9, 22.4, 24.9, 27.4, 29.9, 32.3, 34.8, 37.2, 39.6, 42.1, 44.4, 46.7, 49, 51.3, 53.7, 56, 58.4, 60.7, 63, 65.3, 67.7, 70.2, 72.7, 75.3),
    q1 = c(0.50506, 0.46449, 0.42751, 0.39356, 0.36222, 0.33314, 0.30604, 0.28069, 0.25691, 0.23453, 0.21359, 0.19438, 0.17584, 0.15796, 0.14077, 0.12426, 0.10842, 0.09328, 0.07883, 0.06506, 0.05143, 0.03892, 0.0276, 0.01782, 0.01011),
    q2 = c(0.57037, 0.52896, 0.49039, 0.4543, 0.4204, 0.38844, 0.35822, 0.32958, 0.30238, 0.27648, 0.2513, 0.22804, 0.20561, 0.18401, 0.16324, 0.14329, 0.12414, 0.10558, 0.08773, 0.07087, 0.05534, 0.04135, 0.02896, 0.01847, 0.01037),
    q3 = c(0.59752, 0.55576, 0.51654, 0.47956, 0.44459, 0.41143, 0.37992, 0.34991, 0.32128, 0.29392, 0.26698, 0.24203, 0.21799, 0.19484, 0.17258, 0.15121, 0.13067, 0.11082, 0.09173, 0.07363, 0.05729, 0.04262, 0.02971, 0.01885, 0.01052),
    q5 = c(0.62736, 0.58522, 0.54527, 0.50731, 0.47117, 0.4367, 0.40376, 0.37225, 0.34206, 0.31309, 0.28421, 0.25741, 0.23159, 0.20674, 0.18285, 0.15991, 0.13786, 0.11671, 0.09643, 0.077, 0.05977, 0.04429, 0.03072, 0.01936, 0.01074),
    q10 = c(0.65321, 0.61153, 0.57165, 0.53345, 0.49679, 0.46159, 0.42775, 0.39517, 0.36379, 0.33353, 0.30305, 0.27477, 0.24744, 0.22104, 0.19557, 0.17104, 0.14741, 0.12467, 0.10282, 0.08182, 0.06351, 0.04692, 0.03241, 0.02033, 0.0112),
    q15 = c(0.6644, 0.62306, 0.58336, 0.54519, 0.50846, 0.47308, 0.43897, 0.40605, 0.37427, 0.34357, 0.31267, 0.2839, 0.25602, 0.22903, 0.20295, 0.17776, 0.15345, 0.13003, 0.10748, 0.08578, 0.06675, 0.04944, 0.03424, 0.02153, 0.0119),
    q20 = c(0.68063, 0.63989, 0.60054, 0.56253, 0.52579, 0.49025, 0.45586, 0.42256, 0.3903, 0.35904, 0.32767, 0.2983, 0.26973, 0.24198, 0.21506, 0.18899, 0.16375, 0.13936, 0.11583, 0.09312, 0.0729, 0.05438, 0.038, 0.02416, 0.01354)
  )




  ## TABLE A.I.4. WESTMODEL VALUES FOR MALE PROBABILITIES OFDYING, q(X)
  ## PAGE 63
  west_male <- data.frame(
    e0 = c(18, 20.4, 22.9, 25.3, 27.7, 30.1, 32.5, 34.9, 37.3, 39.7, 42.1, 44.5, 47.1, 49.6, 51.8, 54.1, 56.5, 58.8, 61.2, 63.6, 66, 68.6, 71.2, 73.9, 76.6),
    q1 = c(0.41907, 0.38343, 0.35132, 0.32215, 0.29546, 0.27089, 0.24817, 0.22706, 0.20737, 0.18895, 0.17165, 0.15537, 0.13942, 0.12453, 0.11136, 0.09857, 0.08621, 0.0743, 0.06287, 0.05193, 0.04091, 0.03075, 0.02144, 0.01332, 0.00711),
    q2 = c(0.49692, 0.45848, 0.4231, 0.39033, 0.35985, 0.33135, 0.30463, 0.27948, 0.25575, 0.23329, 0.212, 0.19178, 0.17088, 0.15167, 0.13477, 0.11836, 0.1021, 0.08666, 0.07204, 0.05821, 0.04492, 0.03325, 0.02281, 0.01395, 0.00734),
    q3 = c(0.53102, 0.49135, 0.45454, 0.4202, 0.38805, 0.35783, 0.32936, 0.30244, 0.27693, 0.25272, 0.22968, 0.20772, 0.18466, 0.16356, 0.14502, 0.12708, 0.10944, 0.09264, 0.07668, 0.06153, 0.04715, 0.03469, 0.02364, 0.01434, 0.00748),
    q5 = c(0.56995, 0.52888, 0.49043, 0.45429, 0.42024, 0.38806, 0.35758, 0.32865, 0.30112, 0.27489, 0.24985, 0.22592, 0.20039, 0.17713, 0.15673, 0.13707, 0.11816, 0.09999, 0.08256, 0.06585, 0.05011, 0.03666, 0.02479, 0.0149, 0.00769),
    q10 = c(0.59898, 0.55789, 0.51907, 0.48231, 0.44742, 0.41425, 0.38263, 0.35246, 0.32361, 0.29599, 0.26952, 0.24412, 0.21685, 0.192, 0.17012, 0.14897, 0.12855, 0.10888, 0.08996, 0.07177, 0.05464, 0.03996, 0.02697, 0.01615, 0.00829),
    q15 = c(0.6184, 0.57742, 0.53849, 0.50142, 0.46607, 0.4323, 0.4, 0.36905, 0.33936, 0.31084, 0.28343, 0.25704, 0.22853, 0.20266, 0.17982, 0.15766, 0.13623, 0.11553, 0.09556, 0.07634, 0.05826, 0.04266, 0.02881, 0.01727, 0.00886),
    q20 = c(0.64324, 0.6026, 0.56369, 0.5264, 0.49062, 0.45625, 0.4232, 0.39138, 0.36074, 0.33118, 0.30266, 0.27511, 0.24544, 0.21833, 0.19429, 0.17088, 0.14813, 0.12609, 0.10476, 0.08416, 0.06469, 0.04766, 0.03242, 0.01959, 0.01015)
  )




  ## TABLE A.I.5. NORTH MODEL VALUES FOR FEMALE PROBABILITIES OF DYING. q(X)
  ## PAGE 63
  north_female <- data.frame(
    e0 = c(20, 22.5, 25, 27.5, 30, 32.5, 35, 37.5, 40, 42.5, 45, 47.5, 50, 52.5, 55, 57.5, 60, 62.5, 65, 67.5, 70, 72.5, 75, 77.5, 80),
    q1 = c(0.31973, 0.29202, 0.26715, 0.24461, 0.22405, 0.20517, 0.18774, 0.17158, 0.15653, 0.14247, 0.1293, 0.11695, 0.10549, 0.09502, 0.08488, 0.07508, 0.06565, 0.05663, 0.04801, 0.03982, 0.03205, 0.02465, 0.01823, 0.01218, 0.00781),
    q2 = c(0.40293, 0.37069, 0.34121, 0.31408, 0.28896, 0.2656, 0.24378, 0.22332, 0.20409, 0.18595, 0.16881, 0.15261, 0.13681, 0.1219, 0.10753, 0.09395, 0.08113, 0.06904, 0.05764, 0.04689, 0.03677, 0.02713, 0.01964, 0.01287, 0.00813),
    q3 = c(0.45415, 0.41911, 0.3868, 0.35684, 0.32892, 0.3028, 0.27827, 0.25517, 0.23336, 0.21271, 0.19313, 0.17456, 0.15609, 0.1386, 0.12187, 0.10612, 0.09129, 0.07733, 0.06417, 0.05178, 0.04007, 0.02889, 0.02065, 0.01338, 0.00837),
    q5 = c(0.52217, 0.48342, 0.44735, 0.41363, 0.38199, 0.3522, 0.32408, 0.29747, 0.27223, 0.24825, 0.22543, 0.20372, 0.18169, 0.16094, 0.14136, 0.1229, 0.1055, 0.08907, 0.07355, 0.05886, 0.04492, 0.0315, 0.02217, 0.01415, 0.00873),
    q10 = c(0.58336, 0.54353, 0.50584, 0.4701, 0.43613, 0.40378, 0.37292, 0.34343, 0.31521, 0.28818, 0.26225, 0.23741, 0.21178, 0.18739, 0.16436, 0.14264, 0.12213, 0.10276, 0.08444, 0.06706, 0.05054, 0.03455, 0.02402, 0.01509, 0.00918),
    q15 = c(0.61121, 0.5713, 0.53325, 0.49691, 0.46215, 0.42885, 0.39692, 0.36627, 0.3368, 0.30846, 0.28117, 0.25493, 0.22788, 0.20191, 0.1773, 0.15401, 0.13197, 0.1111, 0.09132, 0.07253, 0.05463, 0.03728, 0.02573, 0.01607, 0.00971),
    q20 = c(0.63767, 0.59794, 0.55977, 0.52307, 0.48775, 0.45374, 0.42095, 0.38933, 0.3588, 0.32932, 0.30083, 0.27334, 0.24517, 0.21784, 0.1918, 0.16705, 0.14354, 0.12121, 0.09996, 0.07973, 0.06043, 0.04166, 0.0286, 0.01789, 0.0108)
  )




  ## TABLE A.I.6. SOUTH MODEL VALUES FOR FEMALE PROBABILITIES OF DYING. q(X)
  ## PAGE 64
  south_female <- data.frame(
    e0 = c(20, 22.5, 25, 27.5, 30, 32.5, 35, 37.5, 40, 42.5, 45, 47.5, 50, 52.5, 55, 57.5, 60, 62.5, 65, 67.5, 70, 72.5, 75, 77.5, 80),
    q1 = c(0.307, 0.28449, 0.26415, 0.24563, 0.22865, 0.21298, 0.19847, 0.18496, 0.17234, 0.16065, 0.15044, 0.14047, 0.13075, 0.1213, 0.11214, 0.10327, 0.09471, 0.08639, 0.07715, 0.06801, 0.0589, 0.04987, 0.04096, 0.03226, 0.02486),
    q2 = c(0.44469, 0.41367, 0.38496, 0.35824, 0.33324, 0.30978, 0.28769, 0.26682, 0.24705, 0.22823, 0.21077, 0.19397, 0.17781, 0.16226, 0.14732, 0.13295, 0.11899, 0.10546, 0.09199, 0.0793, 0.06717, 0.05564, 0.04472, 0.0345, 0.02614),
    q3 = c(0.50808, 0.47315, 0.44059, 0.41008, 0.3814, 0.35435, 0.32876, 0.3045, 0.28145, 0.25935, 0.23855, 0.2186, 0.19948, 0.18112, 0.16351, 0.14661, 0.13025, 0.11411, 0.09906, 0.08475, 0.07122, 0.05849, 0.04661, 0.03564, 0.02679),
    q5 = c(0.56057, 0.5224, 0.48664, 0.45301, 0.42128, 0.39125, 0.36278, 0.33571, 0.30993, 0.28511, 0.26155, 0.239, 0.21741, 0.19674, 0.17692, 0.15793, 0.13971, 0.12209, 0.10526, 0.08964, 0.07493, 0.06117, 0.04842, 0.03676, 0.02745),
    q10 = c(0.60018, 0.5614, 0.52462, 0.48966, 0.45635, 0.42456, 0.39415, 0.36504, 0.33712, 0.30969, 0.28367, 0.25873, 0.23481, 0.21185, 0.18982, 0.16865, 0.14832, 0.12869, 0.11051, 0.09359, 0.07778, 0.06311, 0.04965, 0.03746, 0.02783),
    q15 = c(0.61944, 0.5806, 0.54354, 0.50812, 0.47421, 0.44169, 0.41407, 0.38046, 0.35158, 0.32315, 0.29599, 0.2699, 0.24483, 0.22073, 0.19756, 0.17529, 0.15385, 0.13312, 0.11408, 0.09637, 0.07985, 0.06458, 0.05063, 0.03805, 0.02817),
    q20 = c(0.64448, 0.60571, 0.56844, 0.53256, 0.49799, 0.46464, 0.43246, 0.40137, 0.37132, 0.34185, 0.31318, 0.28556, 0.25896, 0.23334, 0.20866, 0.18489, 0.16198, 0.13978, 0.11924, 0.10038, 0.08285, 0.06672, 0.05205, 0.03891, 0.02867)
  )



  ## TABLE A.I.7. EAST MODEL VALUES FOR FEMALE PROBABILITIES OF DYING. q(X)
  ## PAGE 64
  east_female <- data.frame(
    e0 = c(20, 22.5, 25, 27.5, 30, 32.5, 35, 37.5, 40, 42.5, 45, 47.5, 50, 52.5, 55, 57.5, 60, 62.5, 65, 67.5, 70, 72.5, 75, 77.5, 80),
    q1 = c(0.42785, 0.3933, 0.3618, 0.33288, 0.30618, 0.28141, 0.25833, 0.23674, 0.21648, 0.19742, 0.1796, 0.163, 0.14703, 0.13169, 0.11698, 0.1029, 0.08942, 0.07657, 0.06433, 0.05268, 0.04093, 0.03063, 0.02144, 0.0136, 0.00755),
    q2 = c(0.50168, 0.46469, 0.43029, 0.39814, 0.36796, 0.33955, 0.3127, 0.28728, 0.26315, 0.24019, 0.21787, 0.19683, 0.17668, 0.15741, 0.13897, 0.12135, 0.10431, 0.08809, 0.0727, 0.05811, 0.0445, 0.03284, 0.02266, 0.01418, 0.00777),
    q3 = c(0.53306, 0.49504, 0.45941, 0.42588, 0.39422, 0.36426, 0.33582, 0.30876, 0.28298, 0.25837, 0.23414, 0.21121, 0.18929, 0.16834, 0.14832, 0.1292, 0.11074, 0.09316, 0.07646, 0.06058, 0.04616, 0.03389, 0.02325, 0.01446, 0.00788),
    q5 = c(0.56794, 0.52877, 0.49177, 0.45671, 0.42341, 0.39172, 0.36151, 0.33264, 0.30503, 0.27858, 0.25223, 0.2272, 0.2033, 0.18048, 0.15871, 0.13792, 0.11806, 0.09908, 0.08095, 0.06362, 0.04824, 0.03522, 0.02401, 0.01483, 0.00802),
    q10 = c(0.5993, 0.56, 0.52252, 0.48671, 0.45243, 0.41958, 0.38806, 0.35776, 0.32861, 0.30054, 0.27232, 0.24536, 0.21956, 0.19487, 0.17126, 0.14867, 0.12705, 0.10635, 0.08653, 0.06754, 0.05101, 0.03704, 0.02508, 0.01538, 0.00825),
    q15 = c(0.61558, 0.57636, 0.53877, 0.50269, 0.46803, 0.43468, 0.40256, 0.37159, 0.34172, 0.31287, 0.28388, 0.25599, 0.22923, 0.20357, 0.17899, 0.15543, 0.13284, 0.11119, 0.09043, 0.07051, 0.05316, 0.03852, 0.02602, 0.0159, 0.00849),
    q20 = c(0.63645, 0.59744, 0.55981, 0.5235, 0.48843, 0.45452, 0.42173, 0.38998, 0.35923, 0.32943, 0.29964, 0.27058, 0.24262, 0.21573, 0.1899, 0.16509, 0.14124, 0.11834, 0.09633, 0.07519, 0.05651, 0.0409, 0.02757, 0.01678, 0.00892)
  )




  ## TABLE A.I.8. WEST MODEL VALUES FOR FEMALE PROBABILITIES OF DYING. q(X)
  ## PAGE 65
  west_female <- data.frame(
    e0 = c(20, 22.5, 25, 27.5, 30, 32.5, 35, 37.5, 40, 42.5, 45, 47.5, 50, 52.5, 55, 57.5, 60, 62.5, 65, 67.5, 70, 72.5, 75, 77.5, 80),
    q1 = c(0.36517, 0.33362, 0.30519, 0.27936, 0.25573, 0.23398, 0.21386, 0.19518, 0.17774, 0.16143, 0.14612, 0.13171, 0.11831, 0.10548, 0.09339, 0.08177, 0.07066, 0.06004, 0.04994, 0.04034, 0.03093, 0.02262, 0.01516, 0.00894, 0.00445),
    q2 = c(0.45, 0.41443, 0.38171, 0.35144, 0.32329, 0.297, 0.27235, 0.24916, 0.22729, 0.2066, 0.187, 0.16837, 0.15061, 0.1328, 0.11636, 0.10064, 0.08581, 0.0718, 0.05857, 0.04608, 0.03441, 0.0274, 0.01623, 0.00939, 0.0046),
    q3 = c(0.48801, 0.45064, 0.41601, 0.38375, 0.35357, 0.32524, 0.29885, 0.27335, 0.24949, 0.22685, 0.20532, 0.18481, 0.16508, 0.14504, 0.12676, 0.10934, 0.09291, 0.0774, 0.06276, 0.04891, 0.03615, 0.02575, 0.01679, 0.00963, 0.00467),
    q5 = c(0.53117, 0.49176, 0.45494, 0.42042, 0.38795, 0.3573, 0.32831, 0.30082, 0.2747, 0.24983, 0.22611, 0.20346, 0.18152, 0.15894, 0.13873, 0.11959, 0.10146, 0.08429, 0.06799, 0.05251, 0.0384, 0.02714, 0.01752, 0.00994, 0.00478),
    q10 = c(0.56544, 0.52555, 0.48794, 0.45237, 0.41865, 0.38661, 0.35611, 0.32702, 0.29922, 0.27263, 0.24715, 0.22271, 0.199, 0.17441, 0.15227, 0.13126, 0.11132, 0.09238, 0.07439, 0.05725, 0.04165, 0.02928, 0.01877, 0.01055, 0.00501),
    q15 = c(0.59028, 0.55022, 0.51217, 0.47596, 0.44144, 0.40847, 0.37693, 0.34671, 0.31773, 0.28989, 0.26313, 0.23737, 0.21229, 0.18613, 0.1626, 0.1402, 0.1189, 0.09864, 0.07935, 0.06094, 0.04426, 0.03102, 0.01981, 0.01107, 0.00522),
    q20 = c(0.62507, 0.58051, 0.54214, 0.50535, 0.47002, 0.43606, 0.40339, 0.37192, 0.34158, 0.31231, 0.28404, 0.25673, 0.2301, 0.20251, 0.17716, 0.15297, 0.1299, 0.10789, 0.08689, 0.06683, 0.04842, 0.03386, 0.02154, 0.01197, 0.0056)
  )




  ## TABLE A.I.9. NORTH MODEL VALUES FOR PROBABILITIES OF DYING. q(X), BOTH SEXES COMBINED
  ## PAGE 65
  north_both <- data.frame(
    e0 = c(18.8, 21.2, 23.6, 26.1, 28.6, 31, 33.5, 36, 38.4, 40.9, 43.4, 45.9, 48.3, 50.8, 53.2, 55.7, 58.1, 60.6, 63.1, 65.7, 68.2, 70.7, 73.3, 75.9, 78.6),
    q1 = c(0.34608, 0.3162, 0.28938, 0.26509, 0.24292, 0.22256, 0.20377, 0.18635, 0.17012, 0.15496, 0.14076, 0.12744, 0.11503, 0.10386, 0.093, 0.08249, 0.07235, 0.06262, 0.05331, 0.04445, 0.03602, 0.02806, 0.02094, 0.01417, 0.00922),
    q2 = c(0.42815, 0.3942, 0.36313, 0.33451, 0.308, 0.28333, 0.26026, 0.23863, 0.21829, 0.19909, 0.18095, 0.16379, 0.14701, 0.13158, 0.1167, 0.10215, 0.08837, 0.0754, 0.06317, 0.05167, 0.04086, 0.03067, 0.02247, 0.01493, 0.00958),
    q3 = c(0.47836, 0.44191, 0.40825, 0.37698, 0.34782, 0.3205, 0.29482, 0.27062, 0.24775, 0.22609, 0.20553, 0.18602, 0.16657, 0.14861, 0.13139, 0.11473, 0.09899, 0.08414, 0.07015, 0.05696, 0.04451, 0.03271, 0.02369, 0.01557, 0.00963),
    q5 = c(0.54455, 0.50481, 0.46772, 0.43297, 0.4003, 0.36949, 0.34037, 0.31277, 0.28658, 0.26166, 0.23793, 0.21533, 0.19235, 0.17113, 0.15096, 0.13182, 0.11367, 0.09645, 0.08012, 0.06462, 0.04989, 0.03575, 0.02555, 0.01655, 0.01037),
    q10 = c(0.60282, 0.56241, 0.52408, 0.48765, 0.45296, 0.41987, 0.38826, 0.35802, 0.32904, 0.30127, 0.27459, 0.24902, 0.22271, 0.19815, 0.17474, 0.15248, 0.13131, 0.1112, 0.09208, 0.07388, 0.05653, 0.03985, 0.0282, 0.01804, 0.01115),
    q15 = c(0.62896, 0.58864, 0.55011, 0.51323, 0.4779, 0.444, 0.41144, 0.38016, 0.35005, 0.32106, 0.29313, 0.26625, 0.23862, 0.21264, 0.18779, 0.16407, 0.14147, 0.11991, 0.09938, 0.0798, 0.0611, 0.04309, 0.03041, 0.01939, 0.01195),
    q20 = c(0.65402, 0.61407, 0.57561, 0.53856, 0.50284, 0.46841, 0.43517, 0.40308, 0.37207, 0.34209, 0.31311, 0.28513, 0.25654, 0.22941, 0.20332, 0.17831, 0.15436, 0.13145, 0.10953, 0.08858, 0.06851, 0.04912, 0.03476, 0.02242, 0.01399)
  )




  ## TABLEA.I.IO. SOUTH MODEL VALUES FOR PROBABILITIES OF DYING.q(X), BOTH SEXES COMBINED"
  ## PAGE 66
  south_both <- data.frame(
    e0 = c(19.9, 22.4, 24.8, 27.2, 29.6, 32, 34.4, 36.8, 39.2, 41.5, 43.9, 46.3, 48.7, 51, 53.4, 55.8, 58.1, 60, 63.1, 65.6, 68, 70.5, 73, 75.5, 78),
    q1 = c(0.32162, 0.29818, 0.27701, 0.25772, 0.24004, 0.22373, 0.20862, 0.19455, 0.18141, 0.16929, 0.15862, 0.14821, 0.13806, 0.12819, 0.11861, 0.10934, 0.1004, 0.0917, 0.08178, 0.07215, 0.06256, 0.05304, 0.04363, 0.03442, 0.02658),
    q2 = c(0.45276, 0.42161, 0.39279, 0.36595, 0.34087, 0.31731, 0.29514, 0.2742, 0.25436, 0.23589, 0.21822, 0.21022, 0.18485, 0.16908, 0.15393, 0.13935, 0.12527, 0.11154, 0.09667, 0.08342, 0.07077, 0.05873, 0.04732, 0.03662, 0.02783),
    q3 = c(0.51319, 0.4785, 0.44614, 0.41583, 0.38733, 0.36045, 0.33501, 0.3109, 0.28798, 0.26658, 0.24569, 0.22564, 0.2064, 0.18793, 0.1702, 0.15318, 0.13678, 0.12082, 0.10382, 0.08898, 0.07492, 0.06168, 0.04929, 0.03781, 0.02852),
    q5 = c(0.56335, 0.5257, 0.49042, 0.45723, 0.42589, 0.39624, 0.36811, 0.34136, 0.31588, 0.29205, 0.26849, 0.24592, 0.2243, 0.20358, 0.18371, 0.16466, 0.14639, 0.12871, 0.11011, 0.09401, 0.0788, 0.06451, 0.05123, 0.03902, 0.02924),
    q10 = c(0.60047, 0.56231, 0.52613, 0.49175, 0.45901, 0.42777, 0.3979, 0.36932, 0.34191, 0.31575, 0.28999, 0.26525, 0.24149, 0.21868, 0.19677, 0.1757, 0.15547, 0.13589, 0.11578, 0.09836, 0.08201, 0.06676, 0.0527, 0.0399, 0.02974),
    q15 = c(0.61768, 0.57949, 0.54309, 0.50835, 0.47512, 0.44328, 0.41274, 0.38341, 0.3552, 0.32819, 0.3015, 0.27583, 0.25112, 0.22734, 0.20446, 0.18246, 0.16127, 0.14073, 0.11978, 0.10158, 0.08452, 0.06864, 0.05403, 0.04077, 0.03028),
    q20 = c(0.64117, 0.60305, 0.56648, 0.53133, 0.49752, 0.46495, 0.43356, 0.40327, 0.37402, 0.34606, 0.31809, 0.2911, 0.26506, 0.23993, 0.21571, 0.19235, 0.16983, 0.14795, 0.12559, 0.10626, 0.08815, 0.07135, 0.05593, 0.042, 0.03105)
  )




  ## TABLE A.I.ll. EAST MODELVALUES FORPROBABILITIES OF DYING.q(x), BOTHSEXESCOMBINED"
  ## PAGE 66
  east_both <- data.frame(
    e0 = c(18.7, 21.2, 23.7, 26.2, 28.7, 31.2, 33.6, 36.1, 38.6, 41, 43.5, 45.9, 48.3, 50.7, 53.1, 55.6, 58, 60.4, 62.8, 65.2, 67.6, 70, 72.5, 75, 77.6),
    q1 = c(0.4674, 0.42976, 0.39546, 0.36396, 0.33488, 0.30791, 0.28277, 0.25925, 0.23719, 0.21643, 0.19701, 0.17907, 0.16179, 0.14515, 0.12917, 0.11384, 0.09915, 0.08513, 0.07176, 0.05902, 0.04631, 0.03488, 0.0246, 0.01576, 0.00886),
    q2 = c(0.53686, 0.49761, 0.46107, 0.42691, 0.39482, 0.36459, 0.33602, 0.30895, 0.28324, 0.25878, 0.23499, 0.21282, 0.1915, 0.17103, 0.1514, 0.13259, 0.11447, 0.09705, 0.0804, 0.06465, 0.05005, 0.0372, 0.02589, 0.01638, 0.0091),
    q3 = c(0.56608, 0.52614, 0.48867, 0.45337, 0.42002, 0.38842, 0.35841, 0.32984, 0.3026, 0.27658, 0.25096, 0.227, 0.20399, 0.18191, 0.16075, 0.14047, 0.12095, 0.10221, 0.08428, 0.06726, 0.05186, 0.03836, 0.02656, 0.01671, 0.00923),
    q5 = c(0.59837, 0.55768, 0.51917, 0.48263, 0.44787, 0.41476, 0.38315, 0.35293, 0.324, 0.29626, 0.26861, 0.24267, 0.21779, 0.19393, 0.17107, 0.14918, 0.1282, 0.10811, 0.08888, 0.07047, 0.05415, 0.03987, 0.02745, 0.01715, 0.00941),
    q10 = c(0.62691, 0.58639, 0.54768, 0.51065, 0.47515, 0.4411, 0.40839, 0.37692, 0.34663, 0.31744, 0.28806, 0.26042, 0.23384, 0.20827, 0.18371, 0.16013, 0.13748, 0.11573, 0.09487, 0.07485, 0.05741, 0.0421, 0.02883, 0.01792, 0.00976),
    q15 = c(0.64059, 0.60028, 0.56161, 0.52446, 0.48874, 0.45435, 0.42121, 0.38924, 0.35839, 0.32859, 0.29863, 0.27029, 0.24295, 0.21661, 0.19126, 0.16687, 0.1434, 0.12084, 0.09916, 0.07833, 0.06012, 0.04411, 0.03023, 0.01878, 0.01024),
    q20 = c(0.65908, 0.61918, 0.58067, 0.54349, 0.50757, 0.47282, 0.43921, 0.40667, 0.37514, 0.3446, 0.314, 0.28478, 0.25651, 0.22918, 0.20279, 0.17733, 0.15277, 0.12911, 0.10632, 0.08437, 0.0649, 0.0478, 0.03291, 0.02056, 0.01129)
  )




  ## TABLE A.I.l2. WESTMODEL VALUES FOR PROBABILITIES OFDYING, q(X), BOTH SEXES COMBINEDa
  ## PAGE 67
  west_both <- data.frame(
    e0 = c(19, 21.4, 23.9, 26.4, 28.8, 31.3, 33.7, 36.2, 38.6, 41.1, 43.5, 46, 48.5, 51, 53.4, 55.8, 58.2, 60.6, 63.1, 65.5, 68, 70.5, 73.1, 75.7, 78.3),
    q1 = c(0.39278, 0.35913, 0.32882, 0.30128, 0.27608, 0.25289, 0.23143, 0.21151, 0.19292, 0.17553, 0.1592, 0.14383, 0.12912, 0.11524, 0.10259, 0.09037, 0.07862, 0.06734, 0.05656, 0.04628, 0.03604, 0.02678, 0.01838, 0.01118, 0.00581),
    q2 = c(0.47403, 0.43699, 0.40291, 0.37136, 0.34202, 0.31459, 0.28888, 0.26469, 0.24187, 0.22027, 0.1998, 0.18036, 0.16099, 0.14247, 0.12579, 0.10972, 0.09415, 0.07941, 0.06547, 0.05229, 0.03979, 0.02908, 0.0196, 0.01173, 0.00594),
    q3 = c(0.51004, 0.47149, 0.43575, 0.40242, 0.37123, 0.34193, 0.31433, 0.28825, 0.26354, 0.2401, 0.2178, 0.19654, 0.17511, 0.15453, 0.13611, 0.11843, 0.10138, 0.08521, 0.06989, 0.05537, 0.04178, 0.03033, 0.0203, 0.01204, 0.00611),
    q5 = c(0.55103, 0.51077, 0.47312, 0.43777, 0.40449, 0.37306, 0.3433, 0.31507, 0.28823, 0.26267, 0.23827, 0.21496, 0.19119, 0.16826, 0.14795, 0.12854, 0.11001, 0.09233, 0.07545, 0.05934, 0.0444, 0.03202, 0.02124, 0.01248, 0.00627),
    q10 = c(0.58262, 0.54211, 0.50388, 0.46771, 0.43339, 0.40077, 0.36969, 0.34005, 0.31171, 0.28459, 0.25861, 0.23368, 0.20814, 0.18342, 0.16141, 0.14033, 0.12015, 0.10083, 0.08236, 0.06469, 0.0483, 0.03475, 0.02297, 0.01342, 0.00669),
    q15 = c(0.60468, 0.56415, 0.52565, 0.489, 0.45406, 0.42068, 0.38875, 0.35815, 0.32881, 0.30062, 0.27353, 0.24744, 0.22061, 0.1946, 0.17142, 0.14914, 0.12778, 0.10729, 0.08765, 0.06883, 0.05143, 0.03698, 0.02442, 0.01425, 0.00708),
    q20 = c(0.63218, 0.59182, 0.55318, 0.51613, 0.48057, 0.4464, 0.41354, 0.38189, 0.35139, 0.32198, 0.29358, 0.26614, 0.23796, 0.21061, 0.18593, 0.16214, 0.13924, 0.11721, 0.09604, 0.07571, 0.05675, 0.04093, 0.02711, 0.01587, 0.00793)
  )


  list("north_male" = north_male,
       "south_male" = south_male,
       "east_male" = east_male,
       "west_male" = west_male,

       "north_female" = north_female,
       "south_female" = south_female,
       "east_female" = east_female,
       "west_female" = west_female,

       "north_both" = north_both,
       "south_both" = south_both,
       "east_both" = east_both,
       "west_both" = west_both)
}

