use super::scalar::{Rbool, Scalar};
use super::*;
use extendr_ffi::{dataptr, R_xlen_t, LOGICAL_GET_REGION, SET_INTEGER_ELT, SEXPTYPE};
use std::iter::FromIterator;















#[derive(PartialEq, Clone)]
pub struct Logicals {
    pub(crate) robj: Robj,
}

use SEXPTYPE::LGLSXP;
macros::gen_vector_wrapper_impl!(
    vector_type: Logicals, // Implements for
    scalar_type: Rbool,    // Element type
    primitive_type: i32,   // Raw element type
    r_prefix: LOGICAL,     // `R` functions prefix
    SEXP: LGLSXP,          // `SEXP`
    doc_name: logical,     // Singular type name used in docs
    altrep_constructor: make_altlogical_from_iterator,
);

macros::gen_from_iterator_impl!(
    vector_type: Logicals,
    collect_from_type: bool,
    underlying_type: Rbool,
    SEXP: LGLSXP,
    assignment: |dest: &mut Rbool, val : bool| *dest = val.into()
);

impl Logicals {

    pub fn get_region(&self, index: usize, dest: &mut [Rbool]) -> usize {
        unsafe {
            let ptr: *mut i32 = dest.as_mut_ptr() as *mut i32;
            LOGICAL_GET_REGION(self.get(), index as R_xlen_t, dest.len() as R_xlen_t, ptr) as usize
        }
    }
}


impl Logicals {
    pub fn set_elt(&mut self, index: usize, val: Rbool) {
        single_threaded(|| unsafe {
            SET_INTEGER_ELT(self.get_mut(), index as R_xlen_t, val.inner());
        })
    }
}

impl Deref for Logicals {
    type Target = [Rbool];


    fn deref(&self) -> &Self::Target {
        unsafe {
            let ptr = dataptr(self.get()) as *const Rbool;
            std::slice::from_raw_parts(ptr, self.len())
        }
    }
}

impl DerefMut for Logicals {

    fn deref_mut(&mut self) -> &mut Self::Target {
        unsafe {
            let ptr = dataptr(self.get_mut()) as *mut Rbool;
            std::slice::from_raw_parts_mut(ptr, self.len())
        }
    }
}

impl std::fmt::Debug for Logicals {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
        if self.len() == 1 {
            write!(f, "{:?}", self.elt(0))
        } else {
            f.debug_list().entries(self.iter()).finish()
        }
    }
}

impl TryFrom<Vec<bool>> for Logicals {
    type Error = Error;

    fn try_from(value: Vec<bool>) -> std::result::Result<Self, Self::Error> {
        Ok(Self { robj: value.into() })
    }
}

#[cfg(test)]
mod tests {
    use crate as extendr_api;
    use crate::r;
    use crate::scalar::Rbool;
    use crate::Rinternals;
    use extendr_api::test;
    use extendr_api::Logicals;

    #[test]
    fn from_iterator() {
        test! {
            let vec : Logicals = (0..3).map(|i| i % 2 == 0).collect();
            assert_eq!(vec, Logicals::from_values([true, false, true]));
        }
    }

    #[test]
    fn from_iterator_ref() {
        test! {
            let src = vec![true, false, true];
            let iter = src.iter();
            let vec : Logicals = iter.collect();
            assert_eq!(vec, Logicals::from_values(src));
        }
    }

    #[test]
    fn iter_mut() {
        test! {
            let mut vec = Logicals::from_values([true, false, true]);
            vec.iter_mut().for_each(|v| *v = true.into());
            assert_eq!(vec, Logicals::from_values([true, true, true]));
        }
    }









    #[test]
    fn from_values_short() {
        test! {

            let vec = Logicals::from_values([true, false, true]);
            assert_eq!(vec.is_altrep(), false);
            assert_eq!(r!(vec.clone()), r!([true, false, true]));
            assert_eq!(vec.elt(1), false);
            let mut dest = [false.into(); 2];
            vec.get_region(1, &mut dest);
            assert_eq!(dest, [false, true]);
        }
    }

    #[test]
    fn from_values_altrep() {
        test! {
            let vec = Logicals::from_values_altrep((0..1000000000).map(|_| Rbool::from(true)));
            assert_eq!(vec.is_altrep(), true);
            assert_eq!(vec.elt(12345678), true);
            let mut dest = [false.into(); 2];
            vec.get_region(12345678, &mut dest);
            assert_eq!(dest, [true, true]);
        }
    }

    #[test]
    fn new() {
        test! {
            let vec = Logicals::new(10);
            assert_eq!(vec.is_logical(), true);
            assert_eq!(vec.len(), 10);
        }
    }

    #[test]
    fn test_vec_bool_logicals_conversion() {
        test! {
            let test = vec![false, true, true, false];
            let test_rbool: Vec<Rbool> = test.clone().into_iter().map(|x|x.into()).collect();
            let test_logicals: Logicals = test.try_into().unwrap();
            assert_eq!(test_logicals.robj.as_logical_slice().unwrap(), &test_rbool);
        }
    }
}
