#!/usr/bin/perl

#
# Copyright 2007 Sun Microsystems, Inc.  All rights reserved.
# Use is subject to license terms.
#
# CDDL HEADER START
#
# The contents of this file are subject to the terms of the
# Common Development and Distribution License (the "License").
# You may not use this file except in compliance with the License.
#
# You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
# or http://www.opensolaris.org/os/licensing.
# See the License for the specific language governing permissions
# and limitations under the License.
#
# When distributing Covered Code, include this CDDL HEADER in each
# file and include the License file at usr/src/OPENSOLARIS.LICENSE.
# If applicable, add the following below this CDDL HEADER, with the
# fields enclosed by brackets "[]" replaced with your own identifying
# information: Portions Copyright [yyyy] [name of copyright owner]
#
# CDDL HEADER END
#
#ident	"%Z%%M%	%I%	%E% SMI"

use warnings;
use strict;

use vars qw($script $limit $i);

$script = "libconv_mk_report_bufsize";


# This perl script is used by the sgs/libconv makefile to generate an
# include file named report_bufsize.h. That include file is used to
# generate an error message that tells the programmer the required
# size for a libconv buffer type.
#
# For details of how that generated file is supposed to be used, please
# read the comment that is placed at the top of that file, the text of
# which is found further down in this script.


# Highest value to test for
((scalar(@ARGV) == 1) && (($limit = int($ARGV[0])) > 0)) ||
	die "usage: $script toplimit\n";

open(OFILE, ">report_bufsize.h") ||
	die "$script: Unable to create report_bufsize.h";


print OFILE <<TEXT;
/*
 * This file was generated by $script,
 * from the libconv Makefile.
 *
 * Many of the buffer types defined in sgs/include/conv.h are defined
 * as a fixed integer rather than using the actual size calculation that
 * would take the lengths of all the possible strings into consideration.
 * The code that implements the conversion function does the proper
 * calculation. It is important that these two values be the same.
 *
 * It is done this way because:
 *
 *	(1) The size calculation uses message length values only available
 *		within the file that implements the conversion function.
 *	(2) Separating the size calculation from the code that uses it
 *		would increase the odds of not updating both, and is
 *		therefore more error prone.
 *
 * If the public size in conv.h and the real size computed by the
 * implementing module are not the same, memory corruption or wasted
 * space will result. Therefore, the code is supposed to contain a
 * preprocessor check that refuses to compile if they don't. This
 * is easy to do, but when the sizes do not match, it can be tedious
 * to determine the right size to set the conv.h value to. This file
 * is used to help with that.
 *
 * Example: Suppose the external size declared in conv.h is
 * CONV_DYN_FLAG_BUFSIZE, and the internal computed value based
 * on the actual strings is FLAGSZ. The module would use the following
 * to ensure they match:
 *
 *	#if CONV_DYN_FLAG_BUFSIZE != FLAGSZ
 *	#define REPORT_BUFSIZE FLAGSZ
 *	#include "report_bufsize.h"
 *	#error "CONV_DYN_FLAG_BUFSIZE does not match FLAGSZ"
 *	#endif 
 *
 * In the error case, report_bufsize.h is included. It will use a #warning
 * preprocessor directive to show the current value of the REPORT_BUFSIZE
 * macro. The programmer will therefore see a message giving the correct
 * size of the value that should be defined in conv.h, followed by a
 * #error line telling them which constant is at fault.
 */

#ifndef REPORT_BUFSIZE
#error "REPORT_BUFSIZE must be defined before including report_bufsize.h"
TEXT

for ($i = 1; $i <= $limit; $i++) {
	print OFILE "#elif REPORT_BUFSIZE == $i\n";
	print OFILE "#warning \"The correct buffer size is $i\"\n";
}

print OFILE <<TEXT;
#else
#warning "report_bufsize.h encountered value larger than $limit. Please raise the limit and rerun"
#endif
TEXT

close OFILE;
