/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2008 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

/*
 * This file contains the core logic of nwamd.
 *
 * This functionality is built around state_machine() which consumes an event.
 * The events which are input to this function are generated by either a change
 * in interface state or a signal.  The events which correspond to signals are
 * either external requests to shutdown or a timer event.  The interface events
 * indicate if an interface has acquired a new address (via DHCP) or if a new
 * interface has appeared on the system.  The latter event is used to detect new
 * links.
 *
 * state_machine() calls high level routines in llp.c and interface.c to act on
 * the state of the machine in response to events.
 *
 * This function is called by the main thread in the program with machine_lock
 * held.  This is the only thread that can add or remove interface and LLP
 * structures, and thus it's safe for this function to use those structures
 * without locks.  See also the locking comments in the interface.c and llp.c
 * block comments.
 */

#include <stdarg.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <libsysevent.h>
#include <net/if.h>
#include <net/route.h>
#include <netinet/in.h>
#include <sys/nvpair.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <syslog.h>

#include "defines.h"
#include "structures.h"
#include "functions.h"
#include "variables.h"

void
state_machine(struct np_event *e)
{
	struct interface *evif;
	llp_t *evllp, *prefllp;
	uint64_t flags;
	boolean_t dhcp_restored = B_FALSE;

	dprintf("state_machine(event type: %s, name: %s)",
	    npe_type_str(e->npe_type), STRING(e->npe_name));
	switch (e->npe_type) {
	case EV_TIMER:
		/* Our timer popped; check our dhcp status. */
		if ((evif = get_interface(e->npe_name)) == NULL) {
			dprintf("couldn't find waiting interface; "
			    "ignoring EV_TIMER event");
			break;
		}
		flags = get_ifflags(evif->if_name, evif->if_family);
		if ((!(evif->if_lflags & IF_DHCPSTARTED) &&
		    !(flags & IFF_DHCPRUNNING)) || ((flags & IFF_UP) &&
		    evif->if_ipv4addr != INADDR_ANY)) {
			/*
			 * Either DHCP came up successfully, or we're no
			 * longer trying to do DHCP on this interface;
			 * so no need to worry about the timer expiring.
			 */
			dprintf("timer popped for %s, but dhcp state is okay "
			    "(ifflags 0x%llx)", evif->if_name, flags);
			break;
		}
		if (evif->if_lflags & IF_DHCPFAILED) {
			dprintf("ignoring timer; interface already failed");
			break;
		}
		/*
		 * dhcp has not yet completed; give up on it for
		 * now and, if it is the currently active llp,
		 * switch to the next best.
		 */
		dprintf("giving up on dhcp on %s (ifflags 0x%llx)",
		    evif->if_name, flags);
		evif->if_lflags |= IF_DHCPFAILED;
		if (interface_is_active(evif)) {
			if ((prefllp = llp_best_avail()) != NULL) {
				llp_swap(prefllp, dcTimer);
			} else {
				dprintf("DHCP timed out, but no better link is "
				    "available");
				report_interface_down(evif->if_name, dcTimer);
			}
		} else {
			dprintf("DHCP failed on inactive link");
			report_interface_down(evif->if_name, dcTimer);
		}

		break;

	case EV_NEWAP:
		if ((evllp = llp_lookup(e->npe_name)) == NULL) {
			dprintf("state_machine: no llp for %s; ignoring "
			    "EV_NEWAP event", STRING(e->npe_name));
			break;
		}

		if (evllp == link_layer_profile) {
			llp_reselect();
		} else {
			evllp->llp_waiting = B_FALSE;
			evllp->llp_failed = B_FALSE;
			prefllp = llp_best_avail();
			if (prefllp == NULL) {
				dprintf("new APs on %s, but no best link is "
				    "available", llp_prnm(evllp));
			} else if (prefllp != link_layer_profile) {
				dprintf("state_machine: new APs on link %s "
				    "caused new preferred llp: %s (was %s)",
				    llp_prnm(evllp), llp_prnm(prefllp),
				    llp_prnm(link_layer_profile));
				llp_swap(prefllp, dcNewAP);
			}
		}
		break;

	case EV_LINKDROP:
	case EV_LINKUP:
	case EV_LINKFADE:
	case EV_LINKDISC:
	case EV_USER:
		if ((evif = get_interface(e->npe_name)) == NULL ||
		    (evllp = llp_lookup(e->npe_name)) == NULL) {
			dprintf("state_machine: either no intf (%p) or no llp "
			    "(%p) for %s; ignoring EV_%s event",
			    (void *)evif, (void *)evllp, STRING(e->npe_name),
			    npe_type_str(e->npe_type));
			break;
		}

		if (e->npe_type == EV_LINKUP || e->npe_type == EV_USER)
			evllp->llp_failed = B_FALSE;

		/*
		 * If we're here because wireless has disconnected, then clear
		 * the DHCP failure flag on this interface; this is a "fresh
		 * start" for the interface, so we should retry DHCP.
		 */
		if (e->npe_type == EV_LINKFADE || e->npe_type == EV_LINKDISC)
			evif->if_lflags &= ~IF_DHCPFAILED;

		prefllp = llp_best_avail();
		if (prefllp == NULL) {
			dprintf("state changed on %s, but no best link is "
			    "available", llp_prnm(evllp));
		} else if (prefllp != link_layer_profile) {
			dprintf("state_machine: change in state of link %s "
			    "resulted in new preferred llp: %s (was %s)",
			    llp_prnm(evllp), llp_prnm(prefllp),
			    llp_prnm(link_layer_profile));
			llp_swap(prefllp,
			    e->npe_type == EV_LINKDROP ? dcUnplugged :
			    e->npe_type == EV_LINKFADE ? dcFaded :
			    e->npe_type == EV_LINKDISC ? dcGone :
			    e->npe_type == EV_USER ? dcUser :
			    dcBetter);
		} else if (prefllp == evllp) {
			/*
			 * If this is a negative event on our preferred link,
			 * then we need to pay closer attention.  (We can
			 * ignore negative events on other links.)
			 */
			switch (e->npe_type) {
			case EV_LINKFADE:
			case EV_LINKDISC:
				/*
				 * If the link has faded or disconnected, then
				 * it's a wireless link, and something has gone
				 * wrong with the connection to the AP.  The
				 * above tests mean that we do intend to stay
				 * with this link for now, so we have to
				 * recover it by attempting to reconnect.
				 * Invoking reselect will do that by calling
				 * bringupinterface.
				 */
				dprintf("disconnect on preferred llp; "
				    "attempting reconnect");
				prefllp->llp_waiting = B_TRUE;
				llp_reselect();
				break;
			case EV_LINKDROP:
				/*
				 * If link status has dropped on a wireless
				 * interface, then we need to check whether
				 * we're still connected.  We're probably not,
				 * and this will cause us to attempt
				 * reconnection.
				 */
				if (prefllp->llp_type == IF_WIRELESS)
					wireless_verify(e->npe_name);
				break;
			}
		}
		if (e->npe_type == EV_USER)
			llp_write_changed_priority(evllp);
		break;

	case EV_NEWADDR:
		if ((evif = get_interface(e->npe_name)) == NULL ||
		    (evllp = llp_lookup(e->npe_name)) == NULL) {
			dprintf("state_machine: either no intf (%p) or no llp "
			    "(%p) for %s; ignoring EV_NEWADDR event",
			    (void *)evif, (void *)evllp, STRING(e->npe_name));
			break;
		}
		evllp->llp_failed = B_FALSE;
		if (evllp->llp_ipv4src == IPV4SRC_DHCP) {
			flags = get_ifflags(evif->if_name, evif->if_family);
			if (!(flags & IFF_DHCPRUNNING) || !(flags & IFF_UP) ||
			    evif->if_ipv4addr == INADDR_ANY) {
				/*
				 * We don't have a DHCP lease.  If we used to
				 * have one, then switch to another profile.
				 * Note that if *we* took the interface down
				 * (which happens if this isn't the one
				 * preferred interface), then this doesn't
				 * signal a DHCP failure.
				 */
				if (!(flags & IFF_DHCPRUNNING))
					evif->if_lflags &= ~IF_DHCPSTARTED;
				if (evif->if_lflags & IF_DHCPACQUIRED) {
					evif->if_lflags &= ~IF_DHCPACQUIRED;
					if (interface_is_active(evif)) {
						evif->if_lflags |=
						    IF_DHCPFAILED;
						prefllp = llp_best_avail();
						if (prefllp != NULL) {
							dprintf("DHCP has "
							    "failed, switch "
							    "interfaces");
							llp_swap(prefllp,
							    dcDHCP);
						} else {
							dprintf("DHCP failed, "
							    "but no better link"
							    " is available");
							report_interface_down(
							    evif->if_name,
							    dcDHCP);
						}
					} else {
						dprintf("DHCP not acquired and "
						    "not active");
					}
				}
				break;
			}

			/*
			 * We have a DHCP lease.  If we'd previously failed
			 * to get one, record that DHCP has been restored.
			 */
			evif->if_timer_expire = 0;
			evif->if_lflags |= IF_DHCPACQUIRED;
			if (evif->if_lflags & IF_DHCPFAILED) {
				evif->if_lflags &= ~IF_DHCPFAILED;
				dhcp_restored = B_TRUE;
			}
		}
		if (evllp != link_layer_profile) {
			if (dhcp_restored &&
			    llp_high_pri(evllp, link_layer_profile) == evllp) {
				dprintf("state_machine: dhcp completed on "
				    "higher priority llp (%s); swapping",
				    llp_prnm(evllp));
				llp_swap(evllp, dcBetter);
			} else {
				dprintf("state_machine: newaddr event was for "
				    "%s, not for current active link (%s); "
				    "taking down %s", evllp->llp_lname,
				    llp_prnm(link_layer_profile),
				    evllp->llp_lname);
				takedowninterface(evllp->llp_lname, dcUnwanted);
				break;
			}
		}
		/*
		 * An address has been assigned to the current active link.
		 * Notify the user, and activate the upper layer profile.
		 *
		 * Since other changes to the link (netmask change, broadcast
		 * addr change, etc.) can cause a NEWADDR event (XXX would
		 * be good if our event generator could do a better job
		 * filtering!), only do this if there is not currently an
		 * active ulp.
		 */
		if (!ulp_is_active()) {
			show_if_status(evllp->llp_lname);
			activate_upper_layer_profile(evllp->llp_ipv4src ==
			    IPV4SRC_DHCP, evllp->llp_lname);
		}
		break;

	case EV_ADDIF:
		/* Plumb the interface */
		if (start_child(IFCONFIG, e->npe_name, "plumb", NULL) != 0) {
			syslog(LOG_ERR, "could not plumb interface %s",
			    e->npe_name);
			return;
		}
		report_interface_added(e->npe_name);
		evllp = llp_lookup(e->npe_name);
		/* Add interface to interface list. */
		evif = add_interface(AF_INET, e->npe_name, 0);
		if (evllp == NULL) {
			/*
			 * Create a new llp entry, and add it to llp list
			 * and /etc/nwam/llp. By default, the llp
			 * has a DHCP IPv4 address source, and IPv6 is
			 * used on the link.  We don't plumb the
			 * IPv6 link yet - this is done for us by
			 * bringupinterface().
			 */
			if ((evllp = llp_add(e->npe_name)) != NULL)
				llp_add_file(evllp);
		}

		/*
		 * start_if_info_collect will launch the gather_interface_info
		 * thread, which will start a scan for wireless interfaces, or
		 * start DHCP on wired interfaces.
		 */
		start_if_info_collect(evif, "check");
		break;

	case EV_REMIF:
		/* Unplumb the interface */
		if (start_child(IFCONFIG, e->npe_name, "unplumb", NULL) != 0) {
			syslog(LOG_ERR, "could not unplumb interface %s",
			    e->npe_name);
			return;
		}
		evllp = llp_lookup(e->npe_name);
		remove_interface(e->npe_name);
		if (evllp != NULL) {
			/* Unplumb IPv6 interface if IPv6 is used on the link */
			if (evllp->llp_ipv6onlink) {
				(void) start_child(IFCONFIG, e->npe_name,
				    "inet6", "unplumb", NULL);
			}
			/* If this llp is active, deactivate it. */
			if (evllp == link_layer_profile)
				llp_swap(NULL, dcRemoved);
			llp_delete(evllp);
		}
		report_interface_removed(e->npe_name);
		break;

	case EV_TAKEDOWN:
		takedowninterface(e->npe_name, dcSelect);
		break;

	case EV_RESELECT:
		if (link_layer_profile == NULL &&
		    (prefllp = llp_best_avail()) != NULL) {
			dprintf("reselect: activating LLP %s",
			    llp_prnm(prefllp));
			llp_swap(prefllp, dcNone);
		} else {
			llp_reselect();
		}
		break;

	case EV_DOOR_TIME:
		check_door_life(NSEC_TO_SEC(gethrtime()));
		break;

	case EV_SHUTDOWN:
		/* Cleanup not expecting to see any more events after this */
		cleanup();
		return;
		/* NOTREACHED */

	default:
		dprintf("unknown event");
		break;
	}
}

void
cleanup(void)
{
	terminate_wireless();
	deactivate_upper_layer_profile();
	if (link_layer_profile != NULL)
		takedowninterface(link_layer_profile->llp_lname, dcShutdown);

	/*
	 * Since actions taken in nwamd result in dhcpagent being
	 * launched, it's under our contract.  Thus, it needs to be
	 * stopped when our stop method is executed.  But it needs
	 * to stick around long enough for us to release any leases
	 * we might have; thus, we don't want the stop method to
	 * explicitly kill it.  We do it here, when we know we've
	 * finished any dhcp cleanup that needed to be done.
	 */
	dprintf("killing dhcpagent");
	(void) start_child(PKILL, "-z", zonename, "dhcpagent", NULL);
}
