/*******************************************************************************
 * Copyright (C) 2004-2008 Intel Corp. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  - Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  - Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  - Neither the name of Intel Corp. nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL Intel Corp. OR THE CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *******************************************************************************/

//----------------------------------------------------------------------------
//
//  File:       PTHICommand.h
//
//  Contents:   header file of PTHICommand class
//
//----------------------------------------------------------------------------
#ifndef __PTHI_COMMAND_H__
#define __PTHI_COMMAND_H__

#include "HECIUnix.h"
#include "StatusCodeDefinitions.h"
#include <list>
#include <string>

#define CERT_HASH_MAX_LENGTH                    64
#define CERT_HASH_MAX_NUMBER                    23
#define NET_TLS_CERT_PKI_MAX_SERIAL_NUMS        3
#define NET_TLS_CERT_PKI_MAX_SERIAL_NUM_LENGTH  16
#define MPS_HOSTNAME_LENGTH                     256

/*
* Constants
*/

static const UINT32 IDER_LOG_ENTRIES = 16;

const UINT8 MAJOR_VERSION = 1;
const UINT8 MINOR_VERSION = 1;
const UINT8 AMT_MAJOR_VERSION = 1;
const UINT8 AMT_MINOR_VERSION = 1;

typedef enum _CFG_PROVISIONING_MODE
{
	CFG_PROVISIONING_MODE_NONE = 0,
	CFG_PROVISIONING_MODE_ENTERPRISE,
	CFG_PROVISIONING_MODE_SMALL_BUSINESS,
	CFG_PROVISIONING_MODE_REMOTE_ASSISTANCE
} CFG_PROVISIONING_MODE;

typedef enum _AMT_PROVISIONING_STATE
{
	PROVISIONING_STATE_PRE = 0,
	PROVISIONING_STATE_IN = 1,
	PROVISIONING_STATE_POST = 2
} AMT_PROVISIONING_STATE;

typedef enum _AMT_FEATURE_STATE_REQUEST
{
	REDIRECTION_SESSION = 0,
	SYSTEM_DEFENSE = 1,
	WEB_UI = 2
} AMT_FEATURE_STATE_REQUEST;

typedef enum _AMT_LAST_HOST_RESET_REASON
{
	RemoteControl = 0,
	Other = 1
} AMT_LAST_HOST_RESET_REASON;

typedef enum _AMT_INTERFACE_INDEX
{
	WIRED = 0,
	WIRELESS = 1
} AMT_INTERFACE_INDEX;

typedef enum _AMT_DHCP_IP_ADDRESS
{
	ACTIVE = 1,
	PASSIVE = 2
} AMT_DHCP_IP_MODE;

typedef UINT32 CFG_IPv4_ADDRESS;

static const UINT32 BIOS_VERSION_LEN = 65;
static const UINT32 VERSIONS_NUMBER = 50;
static const UINT32 UNICODE_STRING_LEN = 20;

typedef enum _AMT_PROVISIONING_TLS_MODE
{
	NOT_READY = 0,
	PSK  = 1,
	PKI  = 2
} AMT_PROVISIONING_TLS_MODE;

typedef enum _AMT_RNG_STATUS
{
	RNG_STATUS_EXIST  = 0,
	RNG_STATUS_IN_PROGRESS = 1,
	RNG_STATUS_NOT_EXIST = 2
} AMT_RNG_STATUS;

#pragma pack(1)

typedef struct _AMT_UNICODE_STRING
{
	UINT16 Length;
	UINT8 String[UNICODE_STRING_LEN];
} AMT_UNICODE_STRING;

typedef struct _AMT_VERSION_TYPE
{
	AMT_UNICODE_STRING Description;
	AMT_UNICODE_STRING Version;
} AMT_VERSION_TYPE;

typedef struct _PTHI_VERSION
{
	UINT8 MajorNumber;
	UINT8 MinorNumber;
} PTHI_VERSION;

typedef struct _CODE_VERSIONS
{
	UINT8 BiosVersion[BIOS_VERSION_LEN];
	UINT32 VersionsCount;
	AMT_VERSION_TYPE Versions[VERSIONS_NUMBER];
} CODE_VERSIONS;

typedef struct _COMMAND_FMT
{
	union
	{
		UINT32 val;
		struct
		{
			UINT32 Operation : 23;
			UINT32 IsResponse : 1;
			UINT32 Class : 8;
		} fields;
	} cmd;

} COMMAND_FMT;

typedef struct _AMT_ANSI_STRING
{
	UINT16 Length;
	CHAR  *Buffer;
} AMT_ANSI_STRING;

typedef struct _PTHI_MESSAGE_HEADER
{
	PTHI_VERSION Version;
	UINT16 Reserved;
	COMMAND_FMT Command;
	UINT32 Length;
} PTHI_MESSAGE_HEADER;

typedef struct _PTHI_RESPONSE_MESSAGE_HEADER
{
	PTHI_MESSAGE_HEADER Header;
	AMT_STATUS Status;
} PTHI_RESPONSE_MESSAGE_HEADER;

typedef struct _CFG_GET_CODE_VERSIONS_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	CODE_VERSIONS CodeVersions;
} CFG_GET_CODE_VERSIONS_RESPONSE;

typedef struct _CFG_GET_PROVISIONING_MODE_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	CFG_PROVISIONING_MODE ProvisioningMode;
	AMT_BOOLEAN LegacyMode;
} CFG_GET_PROVISIONING_MODE_RESPONSE;

typedef struct _CFG_GET_PROVISIONING_STATE_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_PROVISIONING_STATE ProvisioningState;
} CFG_GET_PROVISIONING_STATE_RESPONSE;

typedef struct _CFG_GET_FEATURES_STATE_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
	UINT32 RequestID;
} CFG_GET_FEATURES_STATE_REQUEST;

typedef struct _GET_FEATURES_REDIRECTION_SESSION_STATUS
{
	AMT_BOOLEAN IderOpen;
	AMT_BOOLEAN SolOpen;
	AMT_BOOLEAN Reserved;
} GET_FEATURES_REDIRECTION_SESSION_STATUS;

typedef struct _GET_FEATURES_SYSTEM_DEFENSE_STATE_RESPONSE
{
	AMT_BOOLEAN SystemDefenseActivated;
} GET_FEATURES_SYSTEM_DEFENSE_STATUS_RESPONSE;

typedef struct _GET_FEATURES_WEB_UI_STATE_RESPONSE
{
	AMT_BOOLEAN WebUiEnabled;
} GET_FEATURES_WEB_UI_STATUS_RESPONSE;

typedef union _FEATURES_STATUS_DATA
{
	GET_FEATURES_REDIRECTION_SESSION_STATUS rs;
	GET_FEATURES_SYSTEM_DEFENSE_STATUS_RESPONSE sd;
	GET_FEATURES_WEB_UI_STATUS_RESPONSE webUI;
} FEATURES_STATUS_DATA;

typedef struct _CFG_GET_FEATURES_STATE_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	UINT32 RequestID;
	FEATURES_STATUS_DATA Data;
} CFG_GET_FEATURES_STATE_RESPONSE;

typedef struct _CFG_GET_CURRENT_POWER_POLICY_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
} CFG_GET_CURRENT_POWER_POLICY_REQUEST;

typedef struct _CFG_GET_CURRENT_POWER_POLICY_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_ANSI_STRING PolicyName;
} CFG_GET_CURRENT_POWER_POLICY_RESPONSE;

typedef struct _CFG_GET_LAST_HOST_RESET_REASON_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
} CFG_GET_LAST_HOST_RESET_REASON_REQUEST;

typedef struct _CFG_GET_LAST_HOST_RESET_REASON_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	UINT32 Reason;
	UINT32 RemoteControlTimeStamp;
} CFG_GET_LAST_HOST_RESET_REASON_RESPONSE;

typedef struct _LAN_SETTINGS
{
	AMT_BOOLEAN Enabled;
	CFG_IPv4_ADDRESS Ipv4Address;
	AMT_BOOLEAN DhcpEnabled;
	UINT8 DhcpIpMode;
	UINT8 LinkStatus;
	UINT8 MacAddress[6];
} LAN_SETTINGS;

typedef struct _CFG_GET_LAN_INTERFACE_SETTINGS_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
	UINT32 InterfaceIndex;
} CFG_GET_LAN_INTERFACE_SETTINGS_REQUEST;

typedef struct _CFG_GET_LAN_INTERFACE_SETTINGS_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_BOOLEAN Enabled;
	CFG_IPv4_ADDRESS Ipv4Address;
	AMT_BOOLEAN DhcpEnabled;
	UINT8 DhcpIpMode;
	UINT8 LinkStatus;
	UINT8 MacAddress[6];
} CFG_GET_LAN_INTERFACE_SETTINGS_RESPONSE;

typedef struct _CFG_GET_SECURITY_PARAMETERS_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
} CFG_GET_SECURITY_PARAMETERS_REQUEST;

typedef struct _CFG_GET_SECURITY_PARAMETERS_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_BOOLEAN EnterpriseMode;
	AMT_BOOLEAN TLSEnabled;
	AMT_BOOLEAN HWCryptoEnabled;
	AMT_PROVISIONING_STATE ProvisioningState;
	AMT_BOOLEAN NetworkInterfaceEnabled;
	AMT_BOOLEAN SOLEnabled;
	AMT_BOOLEAN IDEREnabled;
	AMT_BOOLEAN FWUpdateEnabled;
	AMT_BOOLEAN LinkIsUp;
	AMT_BOOLEAN Reserved[8];
} CFG_GET_SECURITY_PARAMETERS_RESPONSE;

typedef struct _CFG_GET_DNS_SUFFIX_LIST_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
} CFG_GET_DNS_SUFFIX_LIST_REQUEST;

typedef struct _CFG_GET_DNS_SUFFIX_LIST_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	UINT16 DataLength;
	UINT8 Data[0];
} CFG_GET_DNS_SUFFIX_LIST_RESPONSE;

/**
 * CFG_SET_ENTERPRISE_ACCESS_REQUEST
 *
 * Flags	Bit 0 - If this bit is set then HostIPAddress is IPv6, otherwise HostIPAddress is IPv4 address.
 *          Bits 1..7 - Reserved, should be zero.
 * HostIPAddress	IPv4 / IPv6 address
 * EnterpriseAccess	1 if LMS has access to enterprise network, otherwise 0.
 */
typedef struct _CFG_SET_ENTERPRISE_ACCESS_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
	UINT8 Flags;
	UINT8 HostIPAddress[16];
	UINT8 EnterpriseAccess;
} CFG_SET_ENTERPRISE_ACCESS_REQUEST;

typedef struct _CFG_SET_ENTERPRISE_ACCESS_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
} CFG_SET_ENTERPRISE_ACCESS_RESPONSE;

typedef struct _CFG_OPEN_USER_INITIATED_CONNECTION_REQUEST
{
	PTHI_MESSAGE_HEADER     Header;
} CFG_OPEN_USER_INITIATED_CONNECTION_REQUEST;

typedef struct _CFG_OPEN_USER_INITIATED_CONNECTION_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER     Header;
} CFG_OPEN_USER_INITIATED_CONNECTION_RESPONSE;

typedef struct _CFG_CLOSE_USER_INITIATED_CONNECTION_REQUEST
{
	PTHI_MESSAGE_HEADER     Header;
} CFG_CLOSE_USER_INITIATED_CONNECTION_REQUEST;

typedef struct _CFG_CLOSE_USER_INITIATED_CONNECTION_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER     Header;
} CFG_CLOSE_USER_INITIATED_CONNECTION_RESPONSE;

typedef enum
{
	AMT_NETWORK_CONNECTION_DIRECT = 0,
	AMT_NETWORK_CONNECTION_VPN,
	AMT_NETWORK_CONNECTION_OUTSIDE_ENTERPRISE
} AMT_NETWORK_CONNECTION_STATUS;

typedef enum
{
	REMOTE_ACCESS_CONNECTION_STATUS_NOT_CONNECTED = 0,
	REMOTE_ACCESS_CONNECTION_STATUS_CONNECTING,
	REMOTE_ACCESS_CONNECTION_STATUS_CONNECTED
} REMOTE_ACCESS_CONNECTION_STATUS;

typedef enum
{
	REMOTE_ACCESS_CONNECTION_TRIGGER_USER_INITIATED = 0,
	REMOTE_ACCESS_CONNECTION_TRIGGER_ALERT,
	REMOTE_ACCESS_CONNECTION_TRIGGER_PROVISIONING,
	REMOTE_ACCESS_CONNECTION_TRIGGER_PERIODIC
} REMOTE_ACCESS_CONNECTION_TRIGGER;

typedef struct _CFG_GET_REMOTE_ACCESS_CONNECTION_STATUS_REQUEST
{
	PTHI_MESSAGE_HEADER     Header;
} CFG_GET_REMOTE_ACCESS_CONNECTION_STATUS_REQUEST;

typedef struct _CFG_GET_REMOTE_ACCESS_CONNECTION_STATUS_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER     Header;
	AMT_NETWORK_CONNECTION_STATUS    AmtNetworkConnectionStatus;
	REMOTE_ACCESS_CONNECTION_STATUS  RemoteAccessConnectionStatus;
	REMOTE_ACCESS_CONNECTION_TRIGGER RemoteAccessConnectionTrigger;
	AMT_ANSI_STRING MpsHostname;
} CFG_GET_REMOTE_ACCESS_CONNECTION_STATUS_RESPONSE;

typedef struct _REMOTE_ACCESS_STATUS
{
	AMT_NETWORK_CONNECTION_STATUS    AmtNetworkConnectionStatus;
	REMOTE_ACCESS_CONNECTION_STATUS  RemoteAccessConnectionStatus;
	REMOTE_ACCESS_CONNECTION_TRIGGER RemoteAccessConnectionTrigger;
	AMT_ANSI_STRING MpsHostname;
} REMOTE_ACCESS_STATUS;


typedef UINT8 AMT_UUID[16];

const AMT_UUID AMT_UUID_LINK_STATE =
	{0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x00,
	 0x00, 0x00, 0x00, 0x01};

typedef struct _STATE_DATA
{
	UINT8 LinkStatus; // (0 - down; 1 - up)
	UINT8 HardSKU;
	UINT8 CryptoFuse; // (0 -  disabled; 1 - enabled)
	UINT8 FlashProtaction; // (0 - disabled; 1 - enabled)
	UINT8 LastMEResetReason;
} STATE_DATA;

typedef struct _STATE_GET_AMT_STATE_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
	AMT_UUID StateVariableIdentifier;
} STATE_GET_AMT_STATE_REQUEST;

typedef struct _STATE_GET_AMT_STATE_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_UUID      StateDataIdentifier;
	UINT32        ByteCount;
	STATE_DATA    StateData;
} STATE_GET_AMT_STATE_RESPONSE;


typedef struct _AMT_HASH_HANDLES
{
	UINT32	Length;
	UINT32	Handles[CERT_HASH_MAX_NUMBER];
} AMT_HASH_HANDLES;

typedef struct _CERTHASH_ENTRY
{
	AMT_BOOLEAN     IsDefault;
	AMT_BOOLEAN     IsActive;
	UINT8           CertificateHash[CERT_HASH_MAX_LENGTH];
	UINT8           HashAlgorithm;
	AMT_ANSI_STRING Name;
} CERTHASH_ENTRY;

typedef enum
{
	CERT_HASH_ALGORITHM_MD5 = 0,  // 16 bytes
	CERT_HASH_ALGORITHM_SHA1,     // 20 bytes
	CERT_HASH_ALGORITHM_SHA256,   // 32 bytes
	CERT_HASH_ALGORITHM_SHA512    // 64 bytes
} CERT_HASH_ALGORITHM;

typedef struct
{
	UINT16      Year;
	UINT16      Month;
	UINT16      DayOfWeek;
	UINT16      Day;
	UINT16      Hour;
	UINT16      Minute;
	UINT16      Second;
} TIME_DATE;

typedef struct _AMT_PROV_AUDIT_RECORD
{
	UINT8                ProvisioningTLSMode;
	AMT_BOOLEAN          SecureDNS;
	AMT_BOOLEAN          HostInitiated;
	CERT_HASH_ALGORITHM  SelectedHashType;
	UINT8                SelectedHashData[CERT_HASH_MAX_LENGTH];
	UINT8                CaCertificateSerials[NET_TLS_CERT_PKI_MAX_SERIAL_NUMS*NET_TLS_CERT_PKI_MAX_SERIAL_NUM_LENGTH];
	AMT_BOOLEAN          AdditionalCaSerialNums;
	AMT_BOOLEAN          IsOemDefault;
	AMT_BOOLEAN          IsTimeValid;
	UINT32               ProvServerIP;
	TIME_DATE            TlsStartTime;
	AMT_ANSI_STRING      ProvServerFQDN;
} AMT_PROV_AUDIT_RECORD;

typedef struct _CFG_GENERATE_RNG_SEED_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
} CFG_GENERATE_RNG_SEED_RESPONSE;

typedef struct _CFG_GET_RNG_SEED_STATUS_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_RNG_STATUS RngStatus;
} CFG_GET_RNG_SEED_STATUS_RESPONSE;

typedef struct _CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_BOOLEAN ZeroTouchEnabled;
} CFG_GET_ZERO_TOUCH_ENABLED_RESPONSE;

typedef struct _CFG_GET_PROVISIONING_TLS_MODE_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_PROVISIONING_TLS_MODE ProvisioningTlsMode;
} CFG_GET_PROVISIONING_TLS_MODE_RESPONSE;

typedef struct _CFG_START_CONFIGURATION_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
} CFG_START_CONFIGURATION_RESPONSE;

typedef struct _CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
} CFG_SET_PROVISIONING_SERVER_OTP_RESPONSE;

typedef struct _CFG_SET_DNS_SUFFIX_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
} CFG_SET_DNS_SUFFIX_RESPONSE;

typedef struct _CFG_GET_HASH_HANDLES_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_HASH_HANDLES HashHandles;
} CFG_GET_HASH_HANDLES_RESPONSE;

typedef struct _CFG_GET_CERTHASH_ENTRY_REQUEST
{
	PTHI_MESSAGE_HEADER Header;
	UINT32 HashHandle;
} CFG_GET_CERTHASH_ENTRY_REQUEST;

typedef struct _CFG_GET_CERTHASH_ENTRY_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	CERTHASH_ENTRY Hash;
} CFG_GET_CERTHASH_ENTRY_RESPONSE;

typedef struct _CFG_GET_PKI_FQDN_SUFFIX_RESPONSE
{
	PTHI_RESPONSE_MESSAGE_HEADER Header;
	AMT_ANSI_STRING Suffix;
} CFG_GET_PKI_FQDN_SUFFIX_RESPONSE;

#pragma pack()

class PTHICommand
{
public:
	PTHICommand(bool verbose = false, unsigned long sendTimeout = 2000);
	~PTHICommand();

	AMT_STATUS GetCodeVersions(CODE_VERSIONS &codeVersions);
	AMT_STATUS GetProvisioningMode(CFG_PROVISIONING_MODE &provisioningMode);
	AMT_STATUS GetProvisioningMode(CFG_PROVISIONING_MODE &provisioningMode, AMT_BOOLEAN &legacy);
	AMT_STATUS GetProvisioningState(AMT_PROVISIONING_STATE &state);
	AMT_STATUS GetFeaturesState(UINT32 requestID, AMT_BOOLEAN (&requestStatus)[2]);
	AMT_STATUS GetLastHostResetReason(UINT32 &Reason, UINT32 &RemoteControlTimeStamp);
	AMT_STATUS GetCurrentPowerPolicy(AMT_ANSI_STRING &policyName);
	AMT_STATUS GetLanInterfaceSettings(UINT32 interfaceSettings, LAN_SETTINGS &lanSettings);
	AMT_STATUS GetHeciVersion(HECI_VERSION &hecVersion);
	AMT_STATUS GetTLSEnabled(AMT_BOOLEAN &tlsEnabled);
	AMT_STATUS GetDNSSuffixList(std::list<std::string> &dnsSuffixList);
	AMT_STATUS SetEnterpriseAccess(UINT8 Flags, UINT8 HostIPAddress[16], UINT8 EnterpriseAccess);
	AMT_STATUS GetFWResetReason(UINT8 &MEResetReason);
	AMT_STATUS OpenUserInitiatedConnection();
	AMT_STATUS CloseUserInitiatedConnection();
	AMT_STATUS GetRemoteAccessConnectionStatus(REMOTE_ACCESS_STATUS &remoteAccessStatus);
	AMT_STATUS GenerateRngKey();
	AMT_STATUS GetRngSeedStatus(AMT_RNG_STATUS &rngStatus);
	AMT_STATUS GetZeroTouchEnabled(AMT_BOOLEAN &zeroTouchEnabled);
	AMT_STATUS GetProvisioningTlsMode(AMT_PROVISIONING_TLS_MODE &provisioningTlsMode);
	AMT_STATUS StartConfiguration();
	AMT_STATUS SetProvisioningServerOTP(AMT_ANSI_STRING passwordOTP);
	AMT_STATUS SetDnsSuffix(AMT_ANSI_STRING dnsSuffix);
	AMT_STATUS EnumerateHashHandles(AMT_HASH_HANDLES &hashHandles);
	AMT_STATUS GetCertificateHashEntry(UINT32 hashHandle, CERTHASH_ENTRY &hashEntry);
	AMT_STATUS GetDnsSuffix(AMT_ANSI_STRING &dnsSuffix);

	HECILinux PTHIClient;

private:
	AMT_STATUS _call(const unsigned char *command, UINT32 command_size, UINT8 **readBuffer, UINT32 rcmd, unsigned int expSize = 0);
	AMT_STATUS _verifyResponseHeader(const UINT32 command, const PTHI_MESSAGE_HEADER &response_header, UINT32 response_size);
	AMT_STATUS _verifyCodeVersions(const CFG_GET_CODE_VERSIONS_RESPONSE &response);
	AMT_STATUS _verifyCurrentPowerPolicy(const CFG_GET_CURRENT_POWER_POLICY_RESPONSE &response);
	AMT_STATUS _verifyGetDNSSuffixList(const CFG_GET_DNS_SUFFIX_LIST_RESPONSE &response);
	AMT_STATUS _verifyRemoteAccessConnectionStatus(const CFG_GET_REMOTE_ACCESS_CONNECTION_STATUS_RESPONSE &response);
	AMT_STATUS _verifyHashHandles(const CFG_GET_HASH_HANDLES_RESPONSE &response);
	AMT_STATUS _verifyGetCertificateHashEntry(const CFG_GET_CERTHASH_ENTRY_RESPONSE &response);
	AMT_STATUS _verifyGetDnsSuffix(const CFG_GET_PKI_FQDN_SUFFIX_RESPONSE &response);

	unsigned long m_sendTimeout;
};

/*
* Constants
*/


const UINT32 CODE_VERSIONS_REQUEST     = 0x0400001A;
const UINT32 CODE_VERSIONS_RESPONSE    = 0x0480001A;

const PTHI_MESSAGE_HEADER GET_CODE_VERSION_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{CODE_VERSIONS_REQUEST}}, 0
};

const UINT32 PROVISIONING_MODE_REQUEST     = 0x04000008;
const UINT32 PROVISIONING_MODE_RESPONSE    = 0x04800008;

const PTHI_MESSAGE_HEADER GET_PROVISIONING_MODE_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{PROVISIONING_MODE_REQUEST}}, 0
};

const UINT32 PROVISIONING_STATE_REQUEST    = 0x04000011;
const UINT32 PROVISIONING_STATE_RESPONSE   = 0x04800011;

const PTHI_MESSAGE_HEADER GET_PROVISIONING_STATE_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{PROVISIONING_STATE_REQUEST}}, 0
};

const UINT32 GET_FEATURES_STATE_REQUEST		= 0x04000049;
const UINT32 GET_FEATURES_STATE_RESPONSE	= 0x04800049;

const PTHI_MESSAGE_HEADER GET_FEATURES_STATE_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_FEATURES_STATE_REQUEST}}, (sizeof(CFG_GET_FEATURES_STATE_REQUEST) - sizeof(PTHI_MESSAGE_HEADER))
};

const UINT32 GET_CURRENT_POWER_POLICY_REQUEST	= 0x04000047;
const UINT32 GET_CURRENT_POWER_POLICY_RESPONSE	= 0x04800047;

const PTHI_MESSAGE_HEADER GET_CURRENT_POWER_POLICY_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_CURRENT_POWER_POLICY_REQUEST}}, 0
};

const UINT32 GET_LAST_HOST_RESET_REASON_REQUEST		= 0x0400004A;
const UINT32 GET_LAST_HOST_RESET_REASON_RESPONSE	= 0x0480004A;

const PTHI_MESSAGE_HEADER GET_LAST_HOST_RESET_REASON_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_LAST_HOST_RESET_REASON_REQUEST}}, 0
};

const UINT32 GET_LAN_INTERFACE_SETTINGS_REQUEST		= 0x04000048;
const UINT32 GET_LAN_INTERFACE_SETTINGS_RESPONSE	= 0x04800048;

const PTHI_MESSAGE_HEADER GET_LAN_INTERFACE_SETTINGS_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_LAN_INTERFACE_SETTINGS_REQUEST}}, (sizeof(CFG_GET_LAN_INTERFACE_SETTINGS_REQUEST) - sizeof(PTHI_MESSAGE_HEADER))
};

const UINT32 GET_SECURITY_PARAMETERS_REQUEST	=0x0400001B;
const UINT32 GET_SECURITY_PARAMETERS_RESPONSE	=0x0480001B;

const PTHI_MESSAGE_HEADER GET_SECURITY_PARAMETERS_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_SECURITY_PARAMETERS_REQUEST}}, 0
};

const UINT32 GET_DNS_SUFFIX_LIST_REQUEST        = 0x0400003E;
const UINT32 GET_DNS_SUFFIX_LIST_RESPONSE       = 0x0480003E;

const PTHI_MESSAGE_HEADER GET_DNS_SUFFIX_LIST_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_DNS_SUFFIX_LIST_REQUEST}}, 0
};

const UINT32 SET_ENTERPRISE_ACCESS_REQUEST   = 0x0400003F;
const UINT32 SET_ENTERPRISE_ACCESS_RESPONSE  = 0x0480003F;

const PTHI_MESSAGE_HEADER SET_ENTERPRISE_ACCESS_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{SET_ENTERPRISE_ACCESS_REQUEST}}, (sizeof(CFG_SET_ENTERPRISE_ACCESS_REQUEST) - sizeof(PTHI_MESSAGE_HEADER))
};

const UINT32 OPEN_USER_INITIATED_CONNECTION_REQUEST    = 0x04000044;
const UINT32 OPEN_USER_INITIATED_CONNECTION_RESPONSE   = 0x04800044;

const PTHI_MESSAGE_HEADER OPEN_USER_INITIATED_CONNECTION_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{OPEN_USER_INITIATED_CONNECTION_REQUEST}}, 0
};

const UINT32 CLOSE_USER_INITIATED_CONNECTION_REQUEST   = 0x04000045;
const UINT32 CLOSE_USER_INITIATED_CONNECTION_RESPONSE  = 0x04800045;

const PTHI_MESSAGE_HEADER CLOSE_USER_INITIATED_CONNECTION_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{CLOSE_USER_INITIATED_CONNECTION_REQUEST}}, 0
};

const UINT32 GET_REMOTE_ACCESS_CONNECTION_STATUS_REQUEST       = 0x04000046;
const UINT32 GET_REMOTE_ACCESS_CONNECTION_STATUS_RESPONSE      = 0x04800046;

const PTHI_MESSAGE_HEADER GET_REMOTE_ACCESS_CONNECTION_STATUS_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_REMOTE_ACCESS_CONNECTION_STATUS_REQUEST}}, 0
};

const UINT32 GET_AMT_STATE_REQUEST         = 0x01000001;
const UINT32 GET_AMT_STATE_RESPONSE        = 0x01800001;

const PTHI_MESSAGE_HEADER GET_AMT_STATE_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_AMT_STATE_REQUEST}}, sizeof(AMT_UUID)
};

const UINT32 GENERATE_RNG_SEED_REQUEST    = 0x04000028;
const UINT32 GENERATE_RNG_SEED_RESPONSE   = 0x04800028;

const PTHI_MESSAGE_HEADER GENERATE_RNG_SEED_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GENERATE_RNG_SEED_REQUEST}}, 0
};

const UINT32 GET_RNG_SEED_STATUS_REQUEST    = 0x0400002E;
const UINT32 GET_RNG_SEED_STATUS_RESPONSE   = 0x0480002E;

const PTHI_MESSAGE_HEADER GET_RNG_SEED_STATUS_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_RNG_SEED_STATUS_REQUEST}}, 0
};

const UINT32 GET_ZERO_TOUCH_ENABLED_REQUEST    = 0x04000030;
const UINT32 GET_ZERO_TOUCH_ENABLED_RESPONSE   = 0x04800030;

const PTHI_MESSAGE_HEADER GET_ZERO_TOUCH_ENABLED_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_ZERO_TOUCH_ENABLED_REQUEST}}, 0
};

const UINT32 GET_PROVISIONING_TLS_MODE_REQUEST     = 0x0400002B;
const UINT32 GET_PROVISIONING_TLS_MODE_RESPONSE    = 0x0480002B;

const PTHI_MESSAGE_HEADER GET_PROVISIONING_TLS_MODE_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_PROVISIONING_TLS_MODE_REQUEST}}, 0
};

const UINT32 START_CONFIGURATION_REQUEST     = 0x04000029;
const UINT32 START_CONFIGURATION_RESPONSE    = 0x04800029;

const PTHI_MESSAGE_HEADER START_CONFIGURATION_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{START_CONFIGURATION_REQUEST}}, 0
};

const UINT32 SET_PROVISIONING_SERVER_OTP_REQUEST     = 0x0400002A;
const UINT32 SET_PROVISIONING_SERVER_OTP_RESPONSE    = 0x0480002A;

const UINT32 SET_DNS_SUFFIX_REQUEST     = 0x0400002F;
const UINT32 SET_DNS_SUFFIX_RESPONSE    = 0x0480002F;

const UINT32 ENUMERATE_HASH_HANDLES_REQUEST     = 0x0400002C;
const UINT32 ENUMERATE_HASH_HANDLES_RESPONSE    = 0x0480002C;

const PTHI_MESSAGE_HEADER ENUMERATE_HASH_HANDLES_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{ENUMERATE_HASH_HANDLES_REQUEST}}, 0
};

const UINT32 GET_CERTHASH_ENTRY_REQUEST     = 0x0400002D;
const UINT32 GET_CERTHASH_ENTRY_RESPONSE    = 0x0480002D;

const PTHI_MESSAGE_HEADER GET_CERTHASH_ENTRY_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_CERTHASH_ENTRY_REQUEST}}, sizeof(UINT32)
};

const UINT32 GET_PKI_FQDN_SUFFIX_REQUEST    = 0x04000036;
const UINT32 GET_PKI_FQDN_SUFFIX_RESPONSE   = 0x04800036;

const PTHI_MESSAGE_HEADER GET_PKI_FQDN_SUFFIX_HEADER = {
	{AMT_MAJOR_VERSION, AMT_MINOR_VERSION}, 0, {{GET_PKI_FQDN_SUFFIX_REQUEST}}, 0
};

#endif

